/*
 * Decompiled with CFR 0.152.
 */
package com.dbeaver.lm.api;

import com.dbeaver.lm.api.LMConstants;
import com.dbeaver.lm.api.LMEnvironment;
import com.dbeaver.lm.api.LMException;
import com.dbeaver.lm.api.LMKeyProvider;
import com.dbeaver.lm.api.LMLicense;
import com.dbeaver.lm.api.LMLicenseListener;
import com.dbeaver.lm.api.LMLicenseStatus;
import com.dbeaver.lm.api.LMLicenseType;
import com.dbeaver.lm.api.LMLicenseValidator;
import com.dbeaver.lm.api.LMProduct;
import com.dbeaver.lm.api.LMReleaseInfo;
import com.dbeaver.lm.api.LMStatusDetails;
import com.dbeaver.lm.api.LMSubscription;
import com.dbeaver.lm.api.LMUtils;
import com.dbeaver.lm.api.LMValidateException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jkiss.code.NotNull;
import org.jkiss.code.Nullable;
import org.jkiss.utils.xml.XMLBuilder;
import org.jkiss.utils.xml.XMLException;
import org.jkiss.utils.xml.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class LMLicenseManager {
    private static final Logger log = Logger.getLogger("LMLicenseManager");
    private static final String DEFAULT_PURCHASE_URL = "https://dbeaver.com/buy";
    private static final Path LEGACY_CONFIG_PATH = Path.of(System.getProperty("user.home"), ".jkiss-lm");
    private static final boolean DELETE_EMPTY_LICENSE_FILE = false;
    private final LMEnvironment environment;
    private final LMKeyProvider keyProvider;
    @Nullable
    private final LMLicenseValidator validator;
    private final Map<String, Map<String, LMLicense>> licenseCache = new HashMap<String, Map<String, LMLicense>>();
    private final Map<String, LMSubscription> subscriptionCache = new LinkedHashMap<String, LMSubscription>();
    private final List<LMLicenseListener> licenseListeners = new ArrayList<LMLicenseListener>();
    private Path configPath;
    private static Path licensePath;
    private static final List<Path> licenseSearchPath;
    private static Throwable lastLicenseReadError;

    static {
        licenseSearchPath = new ArrayList<Path>();
    }

    public static Path getLicenseCustomPath() {
        return licensePath;
    }

    public static void setLicenseCustomPath(String path) {
        licensePath = Path.of(path, new String[0]);
    }

    public static void addLicenseSearchPath(Path file) {
        licenseSearchPath.add(file);
    }

    public LMLicenseManager(@NotNull LMEnvironment environment, @NotNull LMKeyProvider keyProvider, @NotNull LMLicenseValidator validator) {
        this(environment, keyProvider, validator, LEGACY_CONFIG_PATH);
    }

    public LMLicenseManager(@NotNull LMEnvironment environment, @NotNull LMKeyProvider keyProvider, @Nullable LMLicenseValidator validator, @NotNull Path configPath) {
        this.environment = environment;
        this.keyProvider = keyProvider;
        this.validator = validator;
        this.configPath = configPath;
    }

    @Nullable
    public static Throwable getLastLicenseReadError() {
        return lastLicenseReadError;
    }

    public Path getConfigPath() {
        return this.configPath;
    }

    public void setConfigPath(@NotNull Path configPath) {
        this.configPath = configPath;
    }

    public static void setLicensePath(Path licensePath) {
        LMLicenseManager.licensePath = licensePath;
    }

    public void addLicenseListener(@NotNull LMLicenseListener listener) {
        boolean add;
        boolean remove = this.licenseListeners.remove(listener);
        if (remove) {
            log.warning(String.format("Removed existing LMLicenseListener %s", listener));
        }
        if (add = this.licenseListeners.add(listener)) {
            log.fine(String.format("Added LMLicenseListener %s", listener));
        } else {
            log.severe(String.format("Failed to add LMLicenseListener %s", listener));
        }
    }

    public void removeLicenseListener(@NotNull LMLicenseListener listener) {
        boolean remove = this.licenseListeners.remove(listener);
        if (remove) {
            log.fine(String.format("Removed LMLicenseListener %s", listener));
        } else {
            log.warning(String.format("Nothing to remove for LMLicenseListener %s", listener));
        }
    }

    private void fireLicenseChanged(@NotNull String id, @NotNull LMLicense[] value) {
        LMLicense[] copy = Arrays.copyOf(value, value.length);
        for (LMLicenseListener listener : this.licenseListeners) {
            try {
                listener.licenseChanged(id, copy);
            }
            catch (Throwable t) {
                String pattern = "Error while processing licenseChanged for %s";
                String message = String.format(pattern, listener);
                log.log(Level.SEVERE, message, t);
            }
        }
    }

    @NotNull
    public LMLicense[] getProductLicenses(@NotNull LMProduct product) {
        String id = product.getId();
        Map<String, LMLicense> licenseMap = this.licenseCache.get(id);
        if (licenseMap != null) {
            return licenseMap.values().toArray(new LMLicense[0]);
        }
        LMLicense[] licenses = this.readProductLicenses(product);
        this.fireLicenseChanged(id, licenses);
        return licenses;
    }

    public LMLicense importLicense(@NotNull LMProduct product, @NotNull String clientId, @NotNull byte[] licenseData) throws LMException {
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        LMLicense license = new LMLicense(licenseData, decryptionKey);
        if (this.validator != null) {
            this.validator.validateLicense(this, clientId, product, license);
            LMLicense updatedLicense = this.findImportedLicenseById(product, license.getLicenseId());
            if (updatedLicense != null && updatedLicense.getLicenseId().equals(license.getLicenseId())) {
                return updatedLicense;
            }
        }
        this.importLicense(product, license, false);
        return license;
    }

    public void importLicense(@NotNull LMProduct product, @NotNull LMLicense license, boolean forceImport) throws LMException {
        if (!forceImport && license.isExpired() && !license.isSubscription()) {
            String exceptionMessageTemplate = "License %s is expired. You can purchase a new license on our website: %s.";
            String purchaseUrl = this.environment.getProductPurchaseURL();
            if (purchaseUrl == null || purchaseUrl.isBlank()) {
                purchaseUrl = DEFAULT_PURCHASE_URL;
            }
            throw new LMException(String.format(exceptionMessageTemplate, license.getLicenseId(), purchaseUrl));
        }
        if (!license.isValidForProduct(product)) {
            String exceptionMessageTemplate = "License %s doesn't match product %s. If you want to access this DBeaver version, you need to extend your license support or purchase a new license.";
            throw new LMException(String.format(exceptionMessageTemplate, license.getProductId(), product.getId()));
        }
        if (!forceImport && !license.isValidFor(product, null, false)) {
            Object exceptionMessageTemplate = LMConstants.OUT_OF_SUPPORT_TEMPLATE;
            String archiveLink = this.environment.getArchiveURL();
            exceptionMessageTemplate = archiveLink == null || archiveLink.isBlank() ? (String)exceptionMessageTemplate + "." : (String)exceptionMessageTemplate + ": " + archiveLink;
            throw new LMException(String.format((String)exceptionMessageTemplate, license.getLicenseId()));
        }
        LMLicense[] currentLicenses = this.getProductLicenses(product);
        LinkedHashMap<String, LMLicense> licenseMap = new LinkedHashMap<String, LMLicense>();
        LMLicense[] lMLicenseArray = currentLicenses;
        int n = currentLicenses.length;
        int n2 = 0;
        while (n2 < n) {
            LMLicense lic = lMLicenseArray[n2];
            licenseMap.put(lic.getLicenseId(), lic);
            ++n2;
        }
        if (license.getLicenseType() == LMLicenseType.TRIAL) {
            lMLicenseArray = currentLicenses;
            n = currentLicenses.length;
            n2 = 0;
            while (n2 < n) {
                LMLicense oldLic = lMLicenseArray[n2];
                if (oldLic.getLicenseType() == LMLicenseType.TRIAL && !oldLic.getLicenseId().equals(license.getLicenseId()) && oldLic.getProductVersion().equalsIgnoreCase(license.getProductVersion())) {
                    String exceptionMessageTemplate = "You can't import a trial license for %s %s more than once.";
                    throw new LMException(String.format(exceptionMessageTemplate, license.getProductId(), license.getProductVersion()));
                }
                if (oldLic.getLicenseType() == LMLicenseType.EAP) {
                    throw new LMException("You can't import a trial license when participating in Early Access Program.");
                }
                ++n2;
            }
        }
        licenseMap.put(license.getLicenseId(), license);
        this.licenseCache.put(product.getId(), licenseMap);
        this.saveProductLicenses(product);
        this.fireLicenseChanged(product.getId(), currentLicenses);
    }

    public void updateSubscription(@NotNull LMProduct product, @NotNull LMSubscription subscription) throws LMException {
        this.subscriptionCache.put(subscription.getLicenseId(), subscription);
        this.saveProductLicenses(product);
    }

    private void saveProductLicenses(@NotNull LMProduct product) throws LMException {
        Path prodLicenseFile = this.getProductLicensesFile(product, false);
        Map<String, LMLicense> cache = this.licenseCache.get(product.getId());
        Path lmDir = prodLicenseFile.getParent();
        if (!Files.exists(lmDir, new LinkOption[0])) {
            try {
                Files.createDirectories(lmDir, new FileAttribute[0]);
            }
            catch (Exception e) {
                log.warning("Can't create directory '" + String.valueOf(lmDir.toAbsolutePath()) + "' " + e.getMessage());
            }
        }
        try {
            Throwable e = null;
            Object var6_9 = null;
            try (OutputStream out = Files.newOutputStream(prodLicenseFile, new OpenOption[0]);){
                XMLBuilder xml = new XMLBuilder(out, "utf-8");
                xml.setBeautify(true);
                Throwable throwable = null;
                Object var10_15 = null;
                try (XMLBuilder.Element el1 = xml.startElement("product");){
                    XMLBuilder.Element el2;
                    Object var16_24;
                    Throwable throwable2;
                    byte[] encodedData;
                    xml.addAttribute("id", product.getId());
                    for (LMLicense license : cache.values()) {
                        encodedData = license.getEncoded();
                        if (encodedData == null) {
                            log.warning("License '" + license.getLicenseId() + "' is not encoded");
                            continue;
                        }
                        throwable2 = null;
                        var16_24 = null;
                        try {
                            el2 = xml.startElement("license");
                            try {
                                xml.addAttribute("type", "standard");
                                xml.addText((CharSequence)Base64.getEncoder().encodeToString(encodedData));
                            }
                            finally {
                                if (el2 != null) {
                                    el2.close();
                                }
                            }
                        }
                        catch (Throwable throwable3) {
                            if (throwable2 == null) {
                                throwable2 = throwable3;
                            } else if (throwable2 != throwable3) {
                                throwable2.addSuppressed(throwable3);
                            }
                            throw throwable2;
                        }
                    }
                    for (LMSubscription subscription : this.subscriptionCache.values()) {
                        encodedData = subscription.getEncoded();
                        if (encodedData == null) {
                            log.warning("Subscription '" + subscription.getLicenseId() + "' is not encoded");
                            continue;
                        }
                        throwable2 = null;
                        var16_24 = null;
                        try {
                            el2 = xml.startElement("subscription");
                            try {
                                xml.addAttribute("type", "standard");
                                xml.addText((CharSequence)Base64.getEncoder().encodeToString(encodedData));
                            }
                            finally {
                                if (el2 != null) {
                                    el2.close();
                                }
                            }
                        }
                        catch (Throwable throwable4) {
                            if (throwable2 == null) {
                                throwable2 = throwable4;
                            } else if (throwable2 != throwable4) {
                                throwable2.addSuppressed(throwable4);
                            }
                            throw throwable2;
                        }
                    }
                }
                catch (Throwable throwable5) {
                    if (throwable == null) {
                        throwable = throwable5;
                    } else if (throwable != throwable5) {
                        throwable.addSuppressed(throwable5);
                    }
                    throw throwable;
                }
                xml.flush();
            }
            catch (Throwable throwable) {
                if (e == null) {
                    e = throwable;
                } else if (e != throwable) {
                    e.addSuppressed(throwable);
                }
                throw e;
            }
        }
        catch (IOException e) {
            throw new LMException("IO error while saving license file", e);
        }
    }

    public void clearLicensesCache() {
        this.licenseCache.clear();
        this.subscriptionCache.clear();
    }

    @NotNull
    private LMLicense[] readProductLicenses(@NotNull LMProduct product) {
        this.clearLicensesCache();
        LinkedHashMap<String, LMLicense> licenses = new LinkedHashMap<String, LMLicense>();
        ArrayList<LMSubscription> subscriptions = new ArrayList<LMSubscription>();
        if (licensePath != null) {
            if (!Files.exists(licensePath, new LinkOption[0])) {
                log.warning("License file '" + String.valueOf(licensePath.toAbsolutePath()) + "' doesn't exist");
            } else {
                this.readLicenseFromFile(product, licensePath, licenses);
            }
        } else {
            for (Path licenseFile : licenseSearchPath) {
                if (!Files.exists(licenseFile, new LinkOption[0])) continue;
                this.readLicenseFromFile(product, licenseFile, licenses);
            }
        }
        Path legacyProdLicenseFile = this.getProductLicensesFile(product, true);
        Path modernProdLicenseFile = this.getProductLicensesFile(product, false);
        if (!this.configPath.equals(LEGACY_CONFIG_PATH) && !Files.exists(modernProdLicenseFile, new LinkOption[0])) {
            this.getLicensesFromLMDirectory(product, licenses, subscriptions, legacyProdLicenseFile);
        }
        this.getLicensesFromLMDirectory(product, licenses, subscriptions, modernProdLicenseFile);
        this.licenseCache.put(product.getId(), licenses);
        for (LMSubscription subscription : subscriptions) {
            this.subscriptionCache.put(subscription.getLicenseId(), subscription);
        }
        return licenses.values().toArray(new LMLicense[0]);
    }

    private void getLicensesFromLMDirectory(@NotNull LMProduct product, @NotNull Map<String, LMLicense> licenses, @NotNull List<LMSubscription> subscriptions, @NotNull Path prodLicenseFile) {
        if (Files.exists(prodLicenseFile, new LinkOption[0])) {
            try {
                String licenseType;
                Document document = XMLUtils.parseDocument((Path)prodLicenseFile);
                Element rootElement = document.getDocumentElement();
                if (!"product".equals(rootElement.getTagName()) || !product.getId().equals(rootElement.getAttribute("id"))) {
                    throw new LMException("Bad license file structure");
                }
                for (Element licenseElement : XMLUtils.getChildElementList((Element)rootElement, (String)"license")) {
                    licenseType = licenseElement.getAttribute("type");
                    if (licenseType == null || licenseType.isBlank()) {
                        log.warning("No license type");
                        continue;
                    }
                    String licenseEncoded = XMLUtils.getElementBody((Element)licenseElement);
                    if ("standard".equals(licenseType) && licenseEncoded != null) {
                        LMLicense license = this.readStandardLicense(product, licenseEncoded);
                        if (license == null) continue;
                        licenses.put(license.getLicenseId(), license);
                        continue;
                    }
                    log.warning("Unsupported license type: " + licenseType);
                }
                for (Element subElement : XMLUtils.getChildElementList((Element)rootElement, (String)"subscription")) {
                    licenseType = subElement.getAttribute("type");
                    if (licenseType == null || licenseType.isBlank()) {
                        log.warning("No license type");
                        continue;
                    }
                    String subEncoded = XMLUtils.getElementBody((Element)subElement);
                    if ("standard".equals(licenseType) && subEncoded != null) {
                        LMSubscription subscription = this.readStandardSubscription(product, subEncoded);
                        if (subscription == null) continue;
                        subscriptions.add(subscription);
                        continue;
                    }
                    log.warning("Unsupported subscription type: " + licenseType);
                }
            }
            catch (LMException | XMLException e) {
                log.log(Level.SEVERE, "Error parse product license file '" + String.valueOf(prodLicenseFile), e);
            }
        }
    }

    private void readLicenseFromFile(@NotNull LMProduct product, @NotNull Path path, @NotNull Map<String, LMLicense> licenses) {
        try {
            String licenseEncoded = Files.readString(path);
            LMLicense license = this.readStandardLicense(product, licenseEncoded);
            if (license != null) {
                licenses.put(license.getLicenseId(), license);
            }
        }
        catch (Exception e) {
            log.log(Level.WARNING, "Error loading custom license from " + String.valueOf(path.toAbsolutePath()) + ": " + e.getMessage());
            lastLicenseReadError = e;
        }
    }

    @NotNull
    private Path getProductLicensesFile(@NotNull LMProduct product, boolean legacy) {
        if (!legacy) {
            return this.configPath.resolve(product.getId() + ".lic");
        }
        return LEGACY_CONFIG_PATH.resolve(product.getId() + ".lic");
    }

    public LMLicense readStandardLicense(@NotNull LMProduct product, @NotNull String licenseEncoded) throws LMException {
        byte[] licenseEncrypted;
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        try {
            licenseEncrypted = LMUtils.readEncryptedString(new StringReader(licenseEncoded));
        }
        catch (IOException e) {
            throw new LMException("Error reading license", e);
        }
        LMLicense license = null;
        try {
            license = new LMLicense(licenseEncrypted, decryptionKey);
        }
        catch (LMException e) {
            throw new LMException("Error parsing license", e);
        }
        return license;
    }

    private LMSubscription readStandardSubscription(@NotNull LMProduct product, @NotNull String subEncoded) throws LMException {
        byte[] subEncrypted;
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        try {
            subEncrypted = LMUtils.readEncryptedString(new StringReader(subEncoded));
        }
        catch (IOException e) {
            log.log(Level.SEVERE, "Error reading subscription", e);
            return null;
        }
        try {
            return new LMSubscription(subEncrypted, decryptionKey);
        }
        catch (LMException e) {
            log.log(Level.SEVERE, "Error parsing subscription", e);
            return null;
        }
    }

    @Nullable
    public LMLicense getValidProductLicense(@Nullable String clientId, @NotNull LMProduct product) throws LMValidateException {
        LMLicense[] licenses = this.getProductLicenses(product);
        if (licenses.length == 0) {
            return null;
        }
        LMValidateException validateError = null;
        ArrayList<LMLicense> validLicenses = new ArrayList<LMLicense>();
        LMLicense[] lMLicenseArray = licenses;
        int n = licenses.length;
        int n2 = 0;
        while (n2 < n) {
            block15: {
                boolean canValidateLicense;
                LMLicense license = lMLicenseArray[n2];
                boolean bl = canValidateLicense = clientId != null && this.validator != null && (validLicenses.isEmpty() || license.getLicenseType() != LMLicenseType.TRIAL);
                if (license.isValidFor(product, null, true)) {
                    if (canValidateLicense) {
                        try {
                            this.validator.validateLicense(this, clientId, product, license);
                        }
                        catch (LMValidateException e) {
                            validateError = e;
                            break block15;
                        }
                    }
                    validLicenses.add(license);
                } else if (license.isExpired() && license.getLicenseType() != LMLicenseType.TRIAL && canValidateLicense) {
                    try {
                        this.validator.validateLicense(this, clientId, product, license);
                        license = this.findImportedLicenseById(product, license.getLicenseId());
                        if (license != null && !license.isExpired()) {
                            validLicenses.add(license);
                        }
                    }
                    catch (LMValidateException e) {
                        validateError = e;
                    }
                }
            }
            ++n2;
        }
        if (validLicenses.isEmpty()) {
            if (validateError != null) {
                lastLicenseReadError = validateError;
                throw validateError;
            }
            return null;
        }
        lastLicenseReadError = null;
        validLicenses.sort(Comparator.comparing(LMLicense::getProductVersion).thenComparing(LMLicense::getLicenseStartTime));
        int i = validLicenses.size();
        while (i > 0) {
            LMLicense license = (LMLicense)validLicenses.get(i - 1);
            if (license.getLicenseType() != LMLicenseType.TRIAL) {
                return license;
            }
            --i;
        }
        return (LMLicense)validLicenses.get(validLicenses.size() - 1);
    }

    public boolean hasProductLicense(@NotNull LMProduct product) throws LMValidateException {
        return this.getValidProductLicense(null, product) != null;
    }

    @Nullable
    public LMLicense findImportedLicenseById(@NotNull LMProduct product, @NotNull String licenseId) {
        return Arrays.stream(this.getProductLicenses(product)).filter(license -> license.getLicenseId().equals(licenseId)).findFirst().orElse(null);
    }

    @Nullable
    public LMLicense findTrialLicense(@NotNull String clientId, @NotNull LMProduct product) throws LMValidateException {
        LMLicense[] licenses;
        LMValidateException validateError = null;
        LMLicense[] lMLicenseArray = licenses = this.getProductLicenses(product);
        int n = licenses.length;
        int n2 = 0;
        while (n2 < n) {
            block7: {
                LMLicense license = lMLicenseArray[n2];
                if (license.getLicenseType() == LMLicenseType.TRIAL && license.isValidFor(product, null, false)) {
                    if (this.validator != null) {
                        try {
                            this.validator.validateLicense(this, clientId, product, license);
                        }
                        catch (LMValidateException e) {
                            validateError = e;
                            break block7;
                        }
                    }
                    return license;
                }
            }
            ++n2;
        }
        if (validateError != null) {
            if (validateError.getStatus() == LMLicenseStatus.EXPIRED) {
                return null;
            }
            throw validateError;
        }
        return null;
    }

    public void deleteLicense(@NotNull LMProduct product, @NotNull LMLicense license) throws LMException {
        Map<String, LMLicense> licenseMap = this.licenseCache.get(product.getId());
        if (licenseMap == null) {
            throw new LMException("Internal error: product licenses not found");
        }
        licenseMap.remove(license.getLicenseId());
        this.subscriptionCache.remove(license.getLicenseId());
        this.validator.clearLicenseCache(license.getLicenseId());
        this.saveProductLicenses(product);
        this.fireLicenseChanged(product.getId(), licenseMap.values().toArray(new LMLicense[0]));
    }

    public void refreshLicense(@NotNull LMProduct product, @NotNull LMLicense license) throws LMException {
        Map<String, LMLicense> licenseMap = this.licenseCache.get(product.getId());
        if (licenseMap == null) {
            throw new LMException("Product licenses '" + license.getLicenseId() + "' not found in license manager");
        }
        licenseMap.remove(license.getLicenseId());
        this.subscriptionCache.remove(license.getLicenseId());
        this.validator.clearLicenseCache(license.getLicenseId());
        this.saveProductLicenses(product);
        this.fireLicenseChanged(product.getId(), licenseMap.values().toArray(new LMLicense[0]));
    }

    public void clearLicenseCache() {
        this.validator.clearLicenseCache();
    }

    public void clearLicenseCache(@NotNull LMLicense license) {
        this.validator.clearLicenseCache(license.getLicenseId());
    }

    public void validateLicense(@NotNull String clientId, @NotNull LMProduct product, @NotNull LMLicense license) throws LMValidateException {
        if (this.validator != null) {
            this.validator.validateLicense(this, clientId, product, license);
        }
    }

    public LMSubscription getSubscriptionInfo(@NotNull LMLicense license) {
        return this.subscriptionCache.get(license.getLicenseId());
    }

    @Nullable
    public LMReleaseInfo getReleaseInfo(@NotNull LMLicense license) {
        return this.validator.getValidationReleaseInfo(license);
    }

    @NotNull
    public LMSubscription readSubscriptionFromData(@NotNull LMProduct product, @NotNull byte[] encrypted) throws LMException {
        return new LMSubscription(encrypted, this.keyProvider.getDecryptionKey(product));
    }

    @NotNull
    public String getLicenseValidationStatus(@NotNull LMProduct product, @NotNull LMLicense license) {
        String validationStatus;
        LMStatusDetails licenseStatus = license.getLicenseStatus(product);
        if (licenseStatus.isValid() && (validationStatus = this.validator.getLicenseValidationStatus(license)) != null) {
            return validationStatus;
        }
        return licenseStatus.getMessage();
    }
}

