/*
 * Decompiled with CFR 0.152.
 */
package io.yellowbrick.jdbc.oauth2;

import io.yellowbrick.jdbc.DriverConfiguration;
import io.yellowbrick.jdbc.DriverConstants;
import io.yellowbrick.jdbc.dialog.DeviceCodeDialog;
import io.yellowbrick.jdbc.oauth2.FormParameterEncoder;
import io.yellowbrick.jdbc.oauth2.Token;
import io.yellowbrick.jdbc.web.DeviceCodeServer;
import io.yellowbrick.shaded.org.json.JSONException;
import io.yellowbrick.shaded.org.json.JSONObject;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

public class OAuth2Authorizer
implements DriverConstants {
    private static final String ENCODING_JSON = "application/json";
    private static final String ENCODING_FORM_URLENCODED = "application/x-www-form-urlencoded";
    private final DriverConfiguration driverConfiguration;
    private final String url;
    private final Properties info;
    private final boolean verboseVerbose = System.getenv("YBVERBOSEVERBOSE") != null;
    private final boolean verbose = this.verboseVerbose || System.getenv("YBVERBOSE") != null;

    public OAuth2Authorizer(DriverConfiguration driverConfiguration, String url, Properties info) throws SQLException {
        this.driverConfiguration = new DriverConfiguration(info);
        this.url = url;
        this.info = info;
        if (driverConfiguration.clientId == null) {
            throw new SQLException("Missing required OAuth2 parameter: oauth2ClientId");
        }
        if (driverConfiguration.issuer == null) {
            throw new SQLException("Missing required OAuth2 parameter: oauth2Issuer");
        }
    }

    public Token getOAuth2AccessToken() throws SQLException {
        DeviceCodeServer server = null;
        Runnable dialogDispose = null;
        try {
            Endpoints endpoints = this.getAuthorizationEndpoints();
            HttpClient.Builder clientBuilder = HttpClient.newBuilder();
            SSLContext sslContext = this.createSSLContext();
            clientBuilder.sslContext(sslContext);
            HttpClient httpClient = clientBuilder.build();
            HashMap<String, String> devicePayloadParams = new HashMap<String, String>(Map.of("client_id", this.driverConfiguration.clientId, "scope", this.driverConfiguration.scopes));
            if (this.driverConfiguration.loginHint != null && !this.driverConfiguration.loginHint.isEmpty()) {
                devicePayloadParams.put("login_hint", this.driverConfiguration.loginHint);
            }
            if (this.driverConfiguration.audience != null && !this.driverConfiguration.audience.isEmpty()) {
                devicePayloadParams.put("audience", this.driverConfiguration.audience);
            }
            String devicePayload = FormParameterEncoder.toFormEncoding(devicePayloadParams);
            this.trace("Query device endpoint %s with payload %s\n", endpoints.deviceEndpoint, devicePayload);
            HttpRequest deviceRequest = HttpRequest.newBuilder().uri(URI.create(endpoints.deviceEndpoint)).header("Content-Type", ENCODING_FORM_URLENCODED).header("Accept", ENCODING_JSON).POST(HttpRequest.BodyPublishers.ofString(devicePayload)).build();
            HttpResponse<String> deviceResponse = httpClient.send(deviceRequest, HttpResponse.BodyHandlers.ofString());
            if (deviceResponse.statusCode() != 200) {
                if (deviceResponse.statusCode() == 400) {
                    System.err.println("WARNING: Device login has been disabled for Yellowbrick by your administrator.");
                    System.err.printf("RESPONSE: %s\n", deviceResponse.body());
                    Token token = null;
                    return token;
                }
                throw new SQLException("Invalid response: " + deviceResponse.statusCode() + ", body: " + deviceResponse.body());
            }
            Map<String, Object> deviceContent = OAuth2Authorizer.toJSONResponse(deviceResponse);
            this.trace("Got device auth: %d: %s\n", deviceResponse.statusCode(), deviceContent);
            String userCode = this.requireKey(deviceContent, "user_code");
            String url = (String)(deviceContent.containsKey("verification_uri_complete") ? deviceContent.get("verification_uri_complete") : deviceContent.get("verification_uri"));
            if (url == null) {
                throw new SQLException("Missing verification URI in device authorization response");
            }
            this.trace("Browser URL: %s, user code: %s\n", url, userCode);
            AtomicBoolean cancelled = new AtomicBoolean();
            if (this.driverConfiguration.interactionMode == DriverConfiguration.InteractionMode.CONSOLE) {
                if (!this.driverConfiguration.quiet) {
                    System.out.printf("\nTo authenticate to Yellowbrick, please visit this URL:\n\n    %s\n    and enter code %s\n\n", url, userCode);
                }
            } else if (this.driverConfiguration.interactionMode == DriverConfiguration.InteractionMode.BROWSER) {
                int port = OAuth2Authorizer.getRandomFreePort();
                server = new DeviceCodeServer(port, userCode, url);
                this.trace("Device code server started on http://localhost:%d\n", port);
                OAuth2Authorizer.browse("http://localhost:" + port);
            } else {
                dialogDispose = DeviceCodeDialog.show(url, userCode, ok -> cancelled.set(ok == false));
            }
            String deviceCode = this.requireKey(deviceContent, "device_code");
            int expiresIn = (int)Double.parseDouble(this.requireKey(deviceContent, "expires_in"));
            int interval = (int)Double.parseDouble(this.requireKey(deviceContent, "interval"));
            HashMap<String, String> tokenPayloadParams = new HashMap<String, String>(Map.of("grant_type", "urn:ietf:params:oauth:grant-type:device_code", "device_code", deviceCode, "client_id", this.driverConfiguration.clientId));
            if (this.driverConfiguration.clientSecret != null && !this.driverConfiguration.clientSecret.isEmpty()) {
                tokenPayloadParams.put("client_secret", this.driverConfiguration.clientSecret);
            }
            if (this.driverConfiguration.audience != null && !this.driverConfiguration.audience.isEmpty()) {
                tokenPayloadParams.put("audience", this.driverConfiguration.audience);
            }
            String tokenPayload = FormParameterEncoder.toFormEncoding(tokenPayloadParams);
            String authToken = null;
            String refreshToken = null;
            long expireAt = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(expiresIn);
            this.trace("Token will expire in %d seconds, probing ever %d seconds, at %s\n", expiresIn, interval, Instant.ofEpochMilli(expireAt).toString());
            while (System.currentTimeMillis() < expireAt && !cancelled.get()) {
                this.trace("Query token endpoint %s with payload %s\n", endpoints.tokenEndpoint, tokenPayload);
                HttpRequest tokenRequest = HttpRequest.newBuilder().uri(URI.create(endpoints.tokenEndpoint)).header("Content-Type", ENCODING_FORM_URLENCODED).header("Accept", ENCODING_JSON).POST(HttpRequest.BodyPublishers.ofString(tokenPayload)).build();
                HttpResponse<String> tokenResponse = httpClient.send(tokenRequest, HttpResponse.BodyHandlers.ofString());
                Map<String, Object> tokenContent = OAuth2Authorizer.toJSONResponse(tokenResponse);
                this.trace("Got token auth: %d: %s\n", tokenResponse.statusCode(), tokenContent);
                if (tokenResponse.statusCode() == 200) {
                    authToken = this.driverConfiguration.tokenType == DriverConfiguration.TokenType.ID_TOKEN ? (String)tokenContent.get("id_token") : (String)tokenContent.get("access_token");
                    refreshToken = (String)tokenContent.get("refresh_token");
                    break;
                }
                if (tokenResponse.statusCode() != 400 || !"authorization_pending".equals(tokenContent.get("error") != null ? tokenContent.get("error") : null)) {
                    throw new SQLException("Invalid response: " + tokenResponse.statusCode() + ", error: " + String.valueOf(tokenContent.get("error")) + ", description: " + String.valueOf(tokenContent.get("error_description")));
                }
                TimeUnit.MILLISECONDS.sleep(TimeUnit.SECONDS.toMillis(interval) + 100L);
            }
            if (authToken == null) {
                throw new SQLException(cancelled.get() ? "User cancelled login" : "Device authentication expired");
            }
            this.trace("Returning token: %s\n", authToken);
            Token token = Token.createToken(authToken, refreshToken, this.getTokenExpiration(authToken), this.url, this.info);
            return token;
        }
        catch (SQLException sqlEx) {
            throw sqlEx;
        }
        catch (Exception ex) {
            throw new SQLException("OAuth2 device flow failed", ex);
        }
        finally {
            if (server != null) {
                try {
                    server.stop();
                }
                catch (Exception e) {
                    System.err.println("Failed to stop device code server: " + e.getMessage());
                }
            }
            if (dialogDispose != null) {
                try {
                    dialogDispose.run();
                }
                catch (Exception e) {
                    System.err.println("Failed to stop device code dialog: " + e.getMessage());
                }
            }
        }
    }

    private Instant getTokenExpiration(String authToken) throws SQLException {
        String[] tokenParts = authToken.split("\\.");
        if (tokenParts.length < 3) {
            throw new SQLException("Invalid token format, missing payload");
        }
        String payload = tokenParts[1];
        String decodedPayload = new String(Base64.getUrlDecoder().decode(payload), StandardCharsets.UTF_8);
        Map<String, Object> payloadMap = new JSONObject(decodedPayload).toMap();
        Instant expiresAt = Instant.ofEpochSecond(Long.parseLong(this.requireKey(payloadMap, "exp")));
        this.trace("Token expires at: %s\n", expiresAt);
        return expiresAt;
    }

    public Token refreshOAuth2AccessToken(String refreshToken) throws SQLException {
        try {
            HttpRequest request;
            Endpoints endpoints = this.getAuthorizationEndpoints();
            HashMap<String, String> refreshTokenParams = new HashMap<String, String>(Map.of("grant_type", "refresh_token", "refresh_token", refreshToken, "client_id", this.driverConfiguration.clientId));
            if (this.driverConfiguration.clientSecret != null && !this.driverConfiguration.clientSecret.isEmpty()) {
                refreshTokenParams.put("client_secret", this.driverConfiguration.clientSecret);
            }
            String refreshTokenPayload = FormParameterEncoder.toFormEncoding(refreshTokenParams);
            HttpClient client = this.createHttpClient();
            HttpResponse<String> response = client.send(request = HttpRequest.newBuilder().uri(URI.create(endpoints.tokenEndpoint)).header("Content-Type", ENCODING_FORM_URLENCODED).header("Accept", ENCODING_JSON).POST(HttpRequest.BodyPublishers.ofString(refreshTokenPayload)).build(), HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                throw new SQLException("Failed to refresh token: " + response.statusCode() + " " + response.body());
            }
            Map<String, Object> tokenContent = OAuth2Authorizer.toJSONResponse(response);
            String authToken = this.driverConfiguration.tokenType == DriverConfiguration.TokenType.ID_TOKEN ? (String)tokenContent.get("id_token") : (String)tokenContent.get("access_token");
            String newRefreshToken = (String)tokenContent.getOrDefault("refresh_token", refreshToken);
            return Token.createToken(authToken, newRefreshToken, this.getTokenExpiration(authToken), this.url, this.info);
        }
        catch (SQLException sqlEx) {
            throw sqlEx;
        }
        catch (Exception e) {
            throw new SQLException("Failed to refresh OAuth2 access token", e);
        }
    }

    private static int getRandomFreePort() throws IOException {
        try (ServerSocket socket = new ServerSocket(0);){
            socket.setReuseAddress(true);
            int n = socket.getLocalPort();
            return n;
        }
    }

    protected void trace(String fmt, Object ... args) {
        if (this.verbose) {
            String message = String.format(fmt, args);
            if (!this.verboseVerbose) {
                message = message.replaceAll("(?i)(password|secret)\\s*=\\s*[^&,;\\s]*", "$1=*******");
            }
            System.err.print(message);
        }
    }

    public static void browse(String url) throws Exception {
        ProcessBuilder pb;
        String os = System.getProperty("os.name").toLowerCase();
        if (os.contains("win")) {
            pb = new ProcessBuilder("rundll32", "url.dll,FileProtocolHandler", url);
        } else if (os.contains("mac")) {
            pb = new ProcessBuilder("open", url);
        } else if (os.contains("nix") || os.contains("nux")) {
            pb = new ProcessBuilder("xdg-open", url);
        } else {
            throw new UnsupportedOperationException("Unsupported OS: " + os);
        }
        pb.start();
    }

    public Endpoints getAuthorizationEndpoints() throws Exception {
        HttpRequest request;
        URI configUri = new URI(this.driverConfiguration.issuer + "/.well-known/openid-configuration");
        HttpClient client = this.createHttpClient();
        HttpResponse<String> response = client.send(request = HttpRequest.newBuilder().uri(configUri).GET().header("Accept", ENCODING_JSON).timeout(Duration.ofSeconds(10L)).build(), HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new Exception("Could not retrieve endpoints from " + String.valueOf(configUri) + ", response code: " + response.statusCode());
        }
        Map<String, Object> json = OAuth2Authorizer.toJSONResponse(response);
        Endpoints endpoints = new Endpoints();
        endpoints.tokenEndpoint = this.requireKey(json, "token_endpoint");
        endpoints.deviceEndpoint = this.requireKey(json, "device_authorization_endpoint");
        return endpoints;
    }

    private static Map<String, Object> toJSONResponse(HttpResponse<String> tokenResponse) {
        try {
            return new JSONObject(tokenResponse.body()).toMap();
        }
        catch (JSONException e) {
            HashMap<String, Object> tokenContent = new HashMap<String, Object>();
            tokenContent.put("error", tokenResponse.body());
            tokenContent.put("error_description", "(content returned was not JSON)");
            return tokenContent;
        }
    }

    private String requireKey(Map<String, Object> json, String key) {
        Object value = json.get(key);
        if (value == null) {
            throw new IllegalArgumentException("Missing required key: " + key);
        }
        return value.toString();
    }

    private HttpClient createHttpClient() throws Exception {
        boolean disableTrust = this.driverConfiguration.disableTrust;
        String cacertPath = this.driverConfiguration.cacertPath;
        if (!disableTrust && cacertPath == null) {
            return HttpClient.newHttpClient();
        }
        return HttpClient.newBuilder().sslContext(this.createSSLContext()).build();
    }

    private SSLContext createSSLContext() throws Exception {
        boolean disableTrust = this.driverConfiguration.disableTrust;
        String cacertPath = this.driverConfiguration.cacertPath;
        if (disableTrust) {
            return this.createAllTrustingSSLContext();
        }
        if (cacertPath != null) {
            return this.createCustomCaSslContext(cacertPath);
        }
        return SSLContext.getDefault();
    }

    private SSLContext createAllTrustingSSLContext() throws Exception {
        TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager(){

            @Override
            public void checkClientTrusted(X509Certificate[] chain, String authType) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] chain, String authType) {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return new X509Certificate[0];
            }
        }};
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, trustAllCerts, new SecureRandom());
        return sslContext;
    }

    private SSLContext createCustomCaSslContext(String cacertPath) throws Exception {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        try (FileInputStream caInput = new FileInputStream(cacertPath);){
            X509Certificate caCert = (X509Certificate)cf.generateCertificate(caInput);
            KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
            ks.load(null, null);
            ks.setCertificateEntry("custom-ca", caCert);
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            tmf.init(ks);
            SSLContext sslContext = SSLContext.getInstance("TLS");
            sslContext.init(null, tmf.getTrustManagers(), new SecureRandom());
            SSLContext sSLContext = sslContext;
            return sSLContext;
        }
    }

    private static class Endpoints {
        String tokenEndpoint;
        String deviceEndpoint;

        private Endpoints() {
        }
    }
}

