/*
 * Decompiled with CFR 0.152.
 */
package com.dbeaver.jdbc.files.parquet;

import com.dbeaver.jdbc.files.database.FFSQLType;
import com.dbeaver.jdbc.files.parquet.ParquetValueReader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import org.apache.parquet.example.data.Group;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.schema.GroupType;
import org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.parquet.schema.Type;

public class ParquetConverter {
    private static final int JULIAN_EPOCH_OFFSET_DAYS = 2440588;

    public static ParquetValueReader createValueReader(Type type) {
        return pv -> {
            Group group = pv.group();
            int fieldIndex = pv.fieldIndex();
            return ParquetConverter.convertValue(group, fieldIndex, type);
        };
    }

    private static Object convertSingleValue(Group group, int fieldIndex, Type type, int repetitionIndex) {
        return type.isPrimitive() ? ParquetConverter.convertPrimitiveValue(group, fieldIndex, repetitionIndex, type) : ParquetConverter.convertComplexValue(group, fieldIndex, repetitionIndex, type);
    }

    private static Object convertPrimitiveValue(Group group, int fieldIndex, int repetitionIndex, Type type) {
        PrimitiveType primitive = type.asPrimitiveType();
        FFSQLType jdbcType = ParquetConverter.toSQLType(type);
        PrimitiveType.PrimitiveTypeName primName = primitive.getPrimitiveTypeName();
        switch (jdbcType) {
            case VARCHAR: {
                return group.getBinary(fieldIndex, repetitionIndex).toStringUsingUTF8();
            }
            case DECIMAL: {
                LogicalTypeAnnotation.DecimalLogicalTypeAnnotation decimalAnnotation = (LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)primitive.getLogicalTypeAnnotation();
                int scale = decimalAnnotation.getScale();
                byte[] bytes = group.getBinary(fieldIndex, repetitionIndex).getBytes();
                BigInteger unscaled = new BigInteger(bytes);
                return new BigDecimal(unscaled, scale);
            }
            case DATE: {
                int days = group.getInteger(fieldIndex, repetitionIndex);
                long millis = (long)days * 86400000L;
                return new Date(millis);
            }
            case TIME: 
            case TIME_WITH_TIMEZONE: {
                LogicalTypeAnnotation.TimeLogicalTypeAnnotation timeAnnotation;
                long millis;
                long timeValue;
                LogicalTypeAnnotation logicalTime = primitive.getLogicalTypeAnnotation();
                if (primName == PrimitiveType.PrimitiveTypeName.INT32) {
                    timeValue = group.getInteger(fieldIndex, repetitionIndex);
                } else if (primName == PrimitiveType.PrimitiveTypeName.INT64) {
                    timeValue = group.getLong(fieldIndex, repetitionIndex);
                } else {
                    throw new UnsupportedOperationException("Unsupported primitive type for TIME: " + String.valueOf(primName));
                }
                if (logicalTime instanceof LogicalTypeAnnotation.TimeLogicalTypeAnnotation) {
                    LogicalTypeAnnotation.TimeLogicalTypeAnnotation timeAnnotation2 = (LogicalTypeAnnotation.TimeLogicalTypeAnnotation)logicalTime;
                    switch (timeAnnotation2.getUnit()) {
                        case MILLIS: {
                            v0 = timeValue;
                            break;
                        }
                        case MICROS: {
                            v0 = timeValue / 1000L;
                            break;
                        }
                        default: {
                            v0 = timeValue;
                            break;
                        }
                    }
                } else {
                    v0 = millis = timeValue;
                }
                if (logicalTime instanceof LogicalTypeAnnotation.TimeLogicalTypeAnnotation && !(timeAnnotation = (LogicalTypeAnnotation.TimeLogicalTypeAnnotation)logicalTime).isAdjustedToUTC()) {
                    LocalTime localTime = LocalTime.ofNanoOfDay(millis * 1000000L);
                    ZonedDateTime localZdt = ZonedDateTime.of(LocalDate.ofEpochDay(0L), localTime, ZoneId.systemDefault());
                    millis = localZdt.withZoneSameInstant(ZoneOffset.UTC).toInstant().toEpochMilli();
                }
                return new Time(millis);
            }
            case TIMESTAMP: 
            case TIMESTAMP_WITH_TIMEZONE: {
                if (primName == PrimitiveType.PrimitiveTypeName.INT96) {
                    return ParquetConverter.convertInt96ToTimestamp(group.getInt96(fieldIndex, repetitionIndex));
                }
                long ts = group.getLong(fieldIndex, repetitionIndex);
                LogicalTypeAnnotation logical = primitive.getLogicalTypeAnnotation();
                if (logical instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation) {
                    LogicalTypeAnnotation.TimestampLogicalTypeAnnotation timestampAnnotation = (LogicalTypeAnnotation.TimestampLogicalTypeAnnotation)logical;
                    return switch (timestampAnnotation.getUnit()) {
                        case LogicalTypeAnnotation.TimeUnit.MILLIS -> {
                            if (!timestampAnnotation.isAdjustedToUTC()) {
                                ts = ParquetConverter.adjustLocalMillis(ts);
                            }
                            yield new Timestamp(ts);
                        }
                        case LogicalTypeAnnotation.TimeUnit.MICROS -> ParquetConverter.createTimestamp(ts, timestampAnnotation.isAdjustedToUTC(), 1000L, 1000);
                        case LogicalTypeAnnotation.TimeUnit.NANOS -> ParquetConverter.createTimestamp(ts, timestampAnnotation.isAdjustedToUTC(), 1000000L, 1);
                        default -> throw new IncompatibleClassChangeError();
                    };
                }
                return new Timestamp(ts);
            }
            case BOOLEAN: {
                return group.getBoolean(fieldIndex, repetitionIndex);
            }
            case INTEGER: {
                return group.getInteger(fieldIndex, repetitionIndex);
            }
            case BIGINT: {
                return group.getLong(fieldIndex, repetitionIndex);
            }
            case REAL: {
                return Float.valueOf(group.getFloat(fieldIndex, repetitionIndex));
            }
            case DOUBLE: {
                return group.getDouble(fieldIndex, repetitionIndex);
            }
            case VARBINARY: {
                return group.getBinary(fieldIndex, repetitionIndex).getBytes();
            }
        }
        throw new UnsupportedOperationException("Unsupported JDBC type: " + String.valueOf(jdbcType));
    }

    private static Object convertComplexValue(Group group, int fieldIndex, int repetitionIndex, Type type) {
        LogicalTypeAnnotation logical = type.getLogicalTypeAnnotation();
        if (logical != null) {
            if (logical instanceof LogicalTypeAnnotation.ListLogicalTypeAnnotation) {
                return ParquetConverter.convertList(group, fieldIndex, repetitionIndex);
            }
            if (logical instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation || logical instanceof LogicalTypeAnnotation.MapKeyValueTypeAnnotation) {
                return ParquetConverter.convertMap(group, fieldIndex, repetitionIndex);
            }
        }
        Group nestedGroup = group.getGroup(fieldIndex, repetitionIndex);
        return ParquetConverter.convertGroup(nestedGroup);
    }

    private static Object convertList(Group group, int fieldIndex, int repetitionIndex) {
        Group listGroup = group.getGroup(fieldIndex, repetitionIndex);
        if (listGroup.getType().getFieldCount() == 0) {
            return new ArrayList();
        }
        int elementCount = listGroup.getFieldRepetitionCount(0);
        Type elementType = listGroup.getType().getType(0);
        ArrayList<Object> list = new ArrayList<Object>(elementCount);
        int i = 0;
        while (i < elementCount) {
            list.add(ParquetConverter.convertSingleValue(listGroup, 0, elementType, i));
            ++i;
        }
        return list;
    }

    private static Object convertMap(Group group, int fieldIndex, int repetitionIndex) {
        Group mapGroup = group.getGroup(fieldIndex, repetitionIndex);
        if (mapGroup.getType().getFieldCount() == 0) {
            return new LinkedHashMap();
        }
        int pairCount = mapGroup.getFieldRepetitionCount(0);
        LinkedHashMap<Object, Object> map = new LinkedHashMap<Object, Object>();
        int i = 0;
        while (i < pairCount) {
            Group pair = mapGroup.getGroup(0, i);
            Object key = ParquetConverter.convertValue(pair, 0, pair.getType().getType(0));
            Object value = pair.getFieldRepetitionCount(1) > 0 ? ParquetConverter.convertValue(pair, 1, pair.getType().getType(1)) : null;
            map.put(key, value);
            ++i;
        }
        return map;
    }

    private static Object convertGroup(Group group) {
        LinkedHashMap<String, Object> struct = new LinkedHashMap<String, Object>();
        GroupType groupType = group.getType();
        int i = 0;
        int n = groupType.getFieldCount();
        while (i < n) {
            Type fieldType = groupType.getType(i);
            Object value = ParquetConverter.convertValue(group, i, fieldType);
            struct.put(fieldType.getName(), value);
            ++i;
        }
        return struct;
    }

    private static Timestamp convertInt96ToTimestamp(Binary int96Value) {
        byte[] bytes = int96Value.getBytes();
        ByteBuffer buf = ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN);
        long nanosOfDay = buf.getLong();
        int julianDay = buf.getInt();
        long epochDays = julianDay - 2440588;
        long millisFromDays = epochDays * 86400000L;
        long millisFromNanos = nanosOfDay / 1000000L;
        long totalMillis = millisFromDays + millisFromNanos;
        Timestamp timestamp = new Timestamp(totalMillis);
        int nanos = (int)(nanosOfDay % 1000000000L);
        timestamp.setNanos(nanos);
        return timestamp;
    }

    static FFSQLType toSQLType(Type type) {
        if (type.isPrimitive()) {
            PrimitiveType primitive = type.asPrimitiveType();
            LogicalTypeAnnotation logical = primitive.getLogicalTypeAnnotation();
            if (logical != null) {
                if (logical instanceof LogicalTypeAnnotation.StringLogicalTypeAnnotation || logical instanceof LogicalTypeAnnotation.EnumLogicalTypeAnnotation) {
                    return FFSQLType.VARCHAR;
                }
                if (logical instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
                    return FFSQLType.DECIMAL;
                }
                if (logical instanceof LogicalTypeAnnotation.DateLogicalTypeAnnotation) {
                    return FFSQLType.DATE;
                }
                if (logical instanceof LogicalTypeAnnotation.TimeLogicalTypeAnnotation) {
                    return FFSQLType.TIME;
                }
                if (logical instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation) {
                    return FFSQLType.TIMESTAMP;
                }
                if (logical instanceof LogicalTypeAnnotation.IntervalLogicalTypeAnnotation) {
                    return FFSQLType.NULL;
                }
            }
            return switch (primitive.getPrimitiveTypeName()) {
                case PrimitiveType.PrimitiveTypeName.BOOLEAN -> FFSQLType.BOOLEAN;
                case PrimitiveType.PrimitiveTypeName.INT32 -> FFSQLType.INTEGER;
                case PrimitiveType.PrimitiveTypeName.INT64 -> FFSQLType.BIGINT;
                case PrimitiveType.PrimitiveTypeName.INT96 -> FFSQLType.TIMESTAMP;
                case PrimitiveType.PrimitiveTypeName.FLOAT -> FFSQLType.REAL;
                case PrimitiveType.PrimitiveTypeName.DOUBLE -> FFSQLType.DOUBLE;
                case PrimitiveType.PrimitiveTypeName.BINARY, PrimitiveType.PrimitiveTypeName.FIXED_LEN_BYTE_ARRAY -> FFSQLType.VARBINARY;
                default -> throw new IncompatibleClassChangeError();
            };
        }
        LogicalTypeAnnotation logical = type.getLogicalTypeAnnotation();
        if (logical != null) {
            if (logical instanceof LogicalTypeAnnotation.ListLogicalTypeAnnotation) {
                return FFSQLType.ARRAY;
            }
            if (logical instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation || logical instanceof LogicalTypeAnnotation.MapKeyValueTypeAnnotation) {
                return FFSQLType.STRUCT;
            }
        }
        return FFSQLType.STRUCT;
    }

    static Object convertValue(Group group, int fieldIndex, Type type) {
        int repetitionCount = group.getFieldRepetitionCount(fieldIndex);
        if (repetitionCount == 0) {
            return null;
        }
        if (repetitionCount == 1) {
            return ParquetConverter.convertSingleValue(group, fieldIndex, type, 0);
        }
        ArrayList<Object> list = new ArrayList<Object>(repetitionCount);
        int i = 0;
        while (i < repetitionCount) {
            list.add(ParquetConverter.convertSingleValue(group, fieldIndex, type, i));
            ++i;
        }
        return list;
    }

    private static long adjustLocalMillis(long millis) {
        LocalDateTime local = LocalDateTime.of(1970, 1, 1, 0, 0).plus(millis, ChronoUnit.MILLIS);
        ZonedDateTime zdt = local.atZone(ZoneId.systemDefault());
        return zdt.toInstant().toEpochMilli();
    }

    private static Timestamp createTimestamp(long ts, boolean adjustedToUTC, long divisor, int multiplier) {
        long baseMillis = ts / divisor;
        int remainder = (int)(ts % divisor);
        if (!adjustedToUTC) {
            LocalDateTime local = LocalDateTime.of(1970, 1, 1, 0, 0).plus(baseMillis, ChronoUnit.MILLIS);
            ZonedDateTime zdt = local.atZone(ZoneId.systemDefault());
            baseMillis = zdt.toInstant().toEpochMilli();
        }
        Timestamp timestamp = new Timestamp(baseMillis);
        timestamp.setNanos(timestamp.getNanos() + remainder * multiplier);
        return timestamp;
    }
}

