/*
 * Decompiled with CFR 0.152.
 */
package org.carlspring.cloud.storage.s3fs;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.SequenceInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.carlspring.cloud.storage.s3fs.S3ObjectId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.StorageClass;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;

public final class S3OutputStream
extends OutputStream {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3OutputStream.class);
    protected static final int MIN_UPLOAD_PART_SIZE = 0x500000;
    protected static final int MAX_ALLOWED_UPLOAD_PARTS = 10000;
    private final S3Client s3Client;
    private final S3ObjectId objectId;
    private final StorageClass storageClass;
    private final Map<String, String> metadata;
    private volatile boolean closed;
    private byte[] buffer;
    private int bufferSize;
    private String uploadId;
    private List<String> partETags;

    public S3OutputStream(S3Client s3Client, S3ObjectId objectId) {
        this.s3Client = Objects.requireNonNull(s3Client);
        this.objectId = Objects.requireNonNull(objectId);
        this.metadata = new HashMap<String, String>();
        this.storageClass = null;
    }

    public S3OutputStream(S3Client s3Client, S3ObjectId objectId, StorageClass storageClass) {
        this.s3Client = Objects.requireNonNull(s3Client);
        this.objectId = Objects.requireNonNull(objectId);
        this.storageClass = storageClass;
        this.metadata = new HashMap<String, String>();
    }

    public S3OutputStream(S3Client s3Client, S3ObjectId objectId, Map<String, String> metadata) {
        this.s3Client = Objects.requireNonNull(s3Client);
        this.objectId = Objects.requireNonNull(objectId);
        this.storageClass = null;
        this.metadata = new HashMap<String, String>(metadata);
    }

    public S3OutputStream(S3Client s3Client, S3ObjectId objectId, StorageClass storageClass, Map<String, String> metadata) {
        this.s3Client = Objects.requireNonNull(s3Client);
        this.objectId = Objects.requireNonNull(objectId);
        this.storageClass = storageClass;
        this.metadata = new HashMap<String, String>(metadata);
    }

    protected void setPartETags(List<String> partETags) {
        this.partETags = partETags;
    }

    @Override
    public void write(int bytes) throws IOException {
        this.write(new byte[]{(byte)bytes});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void write(byte[] bytes, int offset, int length) throws IOException {
        if (offset < 0 || offset > bytes.length || length < 0 || offset + length > bytes.length || offset + length < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (length == 0) {
            return;
        }
        if (this.closed) {
            throw new IOException("Already closed");
        }
        S3OutputStream s3OutputStream = this;
        synchronized (s3OutputStream) {
            if (this.uploadId != null && this.partETags.size() >= 10000) {
                throw new IOException("Maximum number of upload parts reached");
            }
            if (length >= 0x500000 || this.bufferSize + length >= 0x500000) {
                this.uploadPart((long)this.bufferSize + (long)length, this.bufferCombinedWith(bytes, offset, length));
                this.bufferSize = 0;
            } else {
                if (this.buffer == null) {
                    this.buffer = new byte[0x500000];
                }
                System.arraycopy(bytes, offset, this.buffer, this.bufferSize, length);
                this.bufferSize += length;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        S3OutputStream s3OutputStream = this;
        synchronized (s3OutputStream) {
            if (this.uploadId == null) {
                this.putObject(this.bufferSize, this.bufferAsStream(), this.getValueFromMetadata("Content-Type"));
                this.buffer = null;
                this.bufferSize = 0;
            } else {
                this.uploadPart(this.bufferSize, this.bufferAsStream());
                this.buffer = null;
                this.bufferSize = 0;
                this.completeMultipartUpload();
            }
            this.closed = true;
        }
    }

    private CreateMultipartUploadResponse createMultipartUpload() throws IOException {
        CreateMultipartUploadRequest.Builder requestBuilder = CreateMultipartUploadRequest.builder().bucket(this.objectId.getBucket()).key(this.objectId.getKey()).metadata(this.metadata);
        if (this.storageClass != null) {
            requestBuilder.storageClass(this.storageClass.toString());
        }
        try {
            return this.s3Client.createMultipartUpload((CreateMultipartUploadRequest)requestBuilder.build());
        }
        catch (SdkException e) {
            throw new IOException("Failed to create S3 client multipart upload", e);
        }
    }

    private void uploadPart(long contentLength, InputStream content) throws IOException {
        if (this.uploadId == null) {
            this.uploadId = this.createMultipartUpload().uploadId();
            if (this.uploadId == null) {
                throw new IOException("Failed to get a valid multipart upload ID from S3 Client");
            }
            this.partETags = new ArrayList<String>();
        }
        int partNumber = this.partETags.size() + 1;
        UploadPartRequest request = (UploadPartRequest)UploadPartRequest.builder().bucket(this.objectId.getBucket()).key(this.objectId.getKey()).uploadId(this.uploadId).partNumber(Integer.valueOf(partNumber)).contentLength(Long.valueOf(contentLength)).build();
        LOGGER.debug("Uploading part {} with length {} for {} ", new Object[]{partNumber, contentLength, this.objectId});
        boolean success = false;
        try {
            try {
                RequestBody requestBody = RequestBody.fromInputStream((InputStream)content, (long)contentLength);
                String partETag = this.s3Client.uploadPart(request, requestBody).eTag();
                LOGGER.debug("Uploaded part {} with length {} for {}", new Object[]{partETag, contentLength, this.objectId});
                this.partETags.add(partETag);
                success = true;
            }
            catch (SdkException e) {
                throw new IOException("Failed to upload multipart data to S3 Client", e);
            }
        }
        finally {
            if (!success) {
                this.closed = true;
                this.abortMultipartUpload();
            }
        }
        if (partNumber >= 10000) {
            this.close();
        }
    }

    private void abortMultipartUpload() {
        LOGGER.debug("Aborting multipart upload {} for {}", (Object)this.uploadId, (Object)this.objectId);
        try {
            AbortMultipartUploadRequest request = (AbortMultipartUploadRequest)AbortMultipartUploadRequest.builder().bucket(this.objectId.getBucket()).key(this.objectId.getKey()).uploadId(this.uploadId).build();
            this.s3Client.abortMultipartUpload(request);
            this.uploadId = null;
            this.partETags = null;
        }
        catch (SdkException e) {
            LOGGER.warn("Failed to abort multipart upload {}: {}", (Object)this.uploadId, (Object)e.getMessage());
        }
    }

    private void completeMultipartUpload() throws IOException {
        int partCount = this.partETags.size();
        LOGGER.debug("Completing upload to {} consisting of {} parts", (Object)this.objectId, (Object)partCount);
        try {
            Collection<CompletedPart> parts = this.buildParts(this.partETags);
            CompletedMultipartUpload completedMultipartUpload = (CompletedMultipartUpload)CompletedMultipartUpload.builder().parts(parts).build();
            CompleteMultipartUploadRequest request = (CompleteMultipartUploadRequest)CompleteMultipartUploadRequest.builder().bucket(this.objectId.getBucket()).key(this.objectId.getKey()).uploadId(this.uploadId).multipartUpload(completedMultipartUpload).build();
            this.s3Client.completeMultipartUpload(request);
        }
        catch (SdkException e) {
            throw new IOException("Failed to complete S3 Client multipart upload", e);
        }
        LOGGER.debug("Completed upload to {} consisting of {} parts", (Object)this.objectId, (Object)partCount);
        this.uploadId = null;
        this.partETags = null;
    }

    private Collection<CompletedPart> buildParts(List<String> partETags) {
        AtomicInteger counter = new AtomicInteger(1);
        return partETags.stream().map(eTag -> (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(counter.getAndIncrement())).eTag(eTag).build()).collect(Collectors.toList());
    }

    private void putObject(long contentLength, InputStream content, String contentType) throws IOException {
        HashMap<String, String> metadataMap = new HashMap<String, String>(this.metadata);
        metadataMap.put("Content-Length", String.valueOf(contentLength));
        PutObjectRequest.Builder requestBuilder = PutObjectRequest.builder().bucket(this.objectId.getBucket()).key(this.objectId.getKey()).contentLength(Long.valueOf(contentLength)).contentType(contentType).metadata(metadataMap);
        if (this.storageClass != null) {
            requestBuilder.storageClass(this.storageClass);
        }
        try {
            RequestBody requestBody = RequestBody.fromInputStream((InputStream)content, (long)contentLength);
            this.s3Client.putObject((PutObjectRequest)requestBuilder.build(), requestBody);
        }
        catch (SdkException e) {
            throw new IOException("Failed to put data into S3 Client object", e);
        }
    }

    private InputStream bufferAsStream() {
        if (this.bufferSize > 0) {
            return new ByteArrayInputStream(this.buffer, 0, this.bufferSize);
        }
        return new InputStream(){

            @Override
            public int read() {
                return -1;
            }
        };
    }

    private InputStream bufferCombinedWith(byte[] bytes, int offset, int length) {
        ByteArrayInputStream stream = new ByteArrayInputStream(bytes, offset, length);
        if (this.bufferSize < 1) {
            return stream;
        }
        return new SequenceInputStream(new ByteArrayInputStream(this.buffer, 0, this.bufferSize), stream);
    }

    private String getValueFromMetadata(String key) {
        if (this.metadata.containsKey(key)) {
            return this.metadata.get(key);
        }
        return null;
    }
}

