const chat = document.getElementById('chat');

const statusBar = document.createElement('div');
statusBar.className = 'status-bar';
statusBar.innerHTML = '<span class="loader"></span><span class="status-bar-text">Waiting for response...</span>';

let copyIcon, executeIcon, editorIcon, infoIcon, cleanIcon;
let copyTooltip, executeTooltip, editorTooltip, cleanTooltip;

document.addEventListener('click', function(event) {
    if (event.target.matches('a.interactive-link')) {
        event.preventDefault();
        const messageId = event.target.getAttribute('data-message-id');
        if (window.executeFunction && messageId) {
            window.executeFunction(parseInt(messageId));
        }
    }
});

marked.setOptions({
    gfm: true,
    breaks: true,
});

marked.use({
    walkTokens(token) {
        if (token.type === 'code' || token.type === 'codespan') return;

        if (token.type === 'text' && typeof token.text === 'string') {
            token.text = token.text.replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
            token.text = token.text.replace(/ {2,}/g, m => '&nbsp;'.repeat(m.length));
        }
    }
});

function initChat(args) {
    copyIcon = args.copy.icon;
    executeIcon = args.execute.icon;
    editorIcon = args.editor.icon;
    infoIcon = args.info.icon;
    cleanIcon = args.clean.icon;

    copyTooltip = args.copy.tooltip;
    executeTooltip = args.execute.tooltip;
    editorTooltip = args.editor.tooltip;
    cleanTooltip = args.clean.tooltip;

}

function addMessage(args) {
    hideCenterText();

    if (document.getElementById(args.id)) return;

    const markdownParsed = marked.parse(args.content);
    const finalContent = parseSqlBlocks(markdownParsed, args.id);

    const div = document.createElement('div');
    div.id = args.id;
    div.className = 'message ' + getClassName(args.role);

    if (args.role === 'error' || args.role === 'warning') {
        const icon = document.createElement('img');
        icon.className = 'message-type-icon';
        icon.src = args.icon;
        icon.alt = args.role;

        div.appendChild(icon);
    }

    const text = document.createElement('p');
    text.innerHTML = finalContent;

    chat.appendChild(div);
    div.appendChild(text);

    const iconsContainer = document.createElement('div');
    iconsContainer.className = 'message-icons-container';

    let cleanIconElement = null;
    if (args.role === 'user') {
        const userMessages = document.querySelectorAll('.message.right');
        if (userMessages.length > 1) {
            cleanIconElement = createCleanIcon(args.id);
            iconsContainer.appendChild(cleanIconElement);
        }
    }

    div.addEventListener('mouseenter', () => {
        iconsContainer.style.opacity = '1';
        iconsContainer.style.visibility = 'visible';
    });

    div.addEventListener('mouseleave', () => {
        iconsContainer.style.opacity = '0';
        iconsContainer.style.visibility = 'hidden';
        hideTooltip();
    });

    const info = createInfoIcon(args);
    iconsContainer.appendChild(info);
    div.appendChild(iconsContainer);

    setTimeout(() => {
        const offset = calculateIconsContainerOffset(div, iconsContainer);
        if (offset > 0) {
            iconsContainer.style.transform = `translateY(-${offset}px) translateX(-5px)`;
        }

    }, 0);

    chat.scrollTop = chat.scrollHeight;
}

function calculateIconsContainerOffset(messageDiv, iconsContainer) {
    const sqlActions = messageDiv.querySelectorAll('.sql-actions');

    if (sqlActions.length === 0) {
        return 0;
    }

    const lastSqlActions = sqlActions[sqlActions.length - 1];
    const sqlButtons = lastSqlActions.querySelectorAll('.sql-button');

    const iconsRect = iconsContainer.getBoundingClientRect();
    let maxOffset = 0;
    let hasCollision = false;

    sqlButtons.forEach(button => {
        const buttonRect = button.getBoundingClientRect();

        const horizontalOverlap = !(iconsRect.right < buttonRect.left || iconsRect.left > buttonRect.right);
        const verticalOverlap = !(iconsRect.bottom < buttonRect.top || iconsRect.top > buttonRect.bottom);

        if (horizontalOverlap && verticalOverlap) {
            hasCollision = true;
            const neededOffset = (iconsRect.bottom - buttonRect.top) + 10;
            maxOffset = Math.max(maxOffset, neededOffset);
        }
    });

    return hasCollision ? maxOffset : 0;
}

function createInfoIcon(args) {
    const info = document.createElement('img');
    info.className = 'message-info-icon';
    info.src = infoIcon;
    info.alt = 'Info';

    info.addEventListener('mouseenter', (event) => {
        showTooltip(event, {
            icon: args.icon,
            tooltipMessage: args.tooltipMessage
        });
    });

    info.addEventListener('mouseleave', () => {
        hideTooltip();
    });

    info.addEventListener('mousemove', (event) => {
        updateTooltipPosition(event);
    });

    return info;
}

function createCleanIcon(messageId) {
    const clean = document.createElement('img');
    clean.className = 'message-clean-icon';
    clean.src = cleanIcon;
    clean.setAttribute('aria-label', cleanTooltip);

    clean.addEventListener('mouseenter', (event) => {
        showCustomTooltip(event, cleanTooltip);
    });

    clean.addEventListener('mouseleave', () => {
        hideCustomTooltip();
    });

    clean.addEventListener('click', (event) => {
        event.stopPropagation();
        clearToHere(messageId);
        hideCustomTooltip();
    });


    return clean;
}

function showCustomTooltip(event, text) {
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip show';
    tooltip.textContent = text;
    tooltip.id = 'custom-clean-tooltip';

    document.body.appendChild(tooltip);

    const rect = tooltip.getBoundingClientRect();
    const viewportHeight = window.innerHeight;

    let x = event.clientX - rect.width/2;
    let y = event.clientY + 15;

    if (x + rect.width > window.innerWidth) {
        x = window.innerWidth - rect.width - 5;
    }

    if (x < 5) {
        x = 5;
    }

    if (y + rect.height > viewportHeight) {
        y = event.clientY - rect.height - 10;
    }

    tooltip.style.left = x + 'px';
    tooltip.style.top = y + 'px';

}

function hideCustomTooltip() {
    const tooltip = document.getElementById('custom-clean-tooltip');
    if (tooltip) {
        tooltip.remove();
    }
}

function addMessageChunk(args) {
    hideCenterText();

    const message = document.getElementById(args.id);
    if (message === undefined) return;

    message.textContent += args.chunk;
    window.scrollTo(0, document.body.scrollHeight);
}

function removeMessage(args) {
    const el = document.getElementById(args.id);
    if (el) el.remove();
}

function clearChat() {
    chat.replaceChildren();
}

function getClassName(role) {
    switch (role) {
        case 'user':
            return 'right';
        case 'function':
        case 'assistant':
            return 'left';
        default:
            return role;
    }
}

function showTooltip(event, messageArgs) {
    hideTooltip();

    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    tooltip.id = 'message-tooltip';
    tooltip.innerHTML = `
        <img src="${messageArgs.icon}" class="tooltip-icon">
        ${messageArgs.tooltipMessage}
    `;

    document.body.appendChild(tooltip);
    updateTooltipPosition(event);
    tooltip.classList.add('show');
}

function hideTooltip() {
    const tooltip = document.getElementById('message-tooltip');
    if (tooltip) {
        tooltip.remove();
    }
}


function updateTooltipPosition(event) {
    const tooltip = document.getElementById('message-tooltip');
    if (!tooltip) return;

    const tooltipRect = tooltip.getBoundingClientRect();
    const viewportHeight = window.innerHeight;

    let x = event.clientX + 5;
    let y = event.clientY - tooltipRect.height - 5;

    if (y < 0) {
        y = event.clientY + 5;
    }

    if (x + tooltipRect.width > window.innerWidth) {
        x = event.clientX - tooltipRect.width - 5;
    }

    if (y + tooltipRect.height > viewportHeight) {
        y = event.clientY - (tooltipRect.height / 2);

        if (y < 0) {
            y = 5;
        }
    }

    tooltip.style.left = x + 'px';
    tooltip.style.top = y + 'px';
}

function parseSqlBlocks(htmlContent, messageId) {
    if (!htmlContent) return '';

    const sqlBlockRegex = /<pre><code class="language-sql">([\s\S]*?)<\/code><\/pre>/g;

    let i = 0
    return htmlContent.replace(sqlBlockRegex, (match, sqlCodeHtml) => {
        i++;
        const sqlCode = decodeHtmlEntities(sqlCodeHtml.trim());
        return createSqlForm(sqlCode, messageId + '.' + i, messageId);
    });
}

function createSqlForm(sqlCode, formId, messageId) {
    const lines = sqlCode.trimEnd().split('<br>');
    const rows = lines.map((line, i) => {
        const numberCell = `<td class="line-number">${i + 1}</td>`;
        const codeCell = `<td class="code-line">${line || '&nbsp;'}</td>`;
        return `<tr>${numberCell}${codeCell}</tr>`;
    }).join('');

    return `
        <div class="sql-form" id="${formId}">
            <div class="sql-content">
                <table class="code-table">
                    <tbody>${rows}</tbody>
                </table>
                <div class="sql-textarea" readonly style="display: none;">${sqlCode}</div>
            </div>
        </div>
        <div class="sql-actions">
            <button class="sql-button sql-execute-btn" onclick="executeInEditor(getSqlCode('${formId}'))" 
                    data-form-id="${formId}" title="${executeTooltip}">
                <img class="sql-button-img" src="${executeIcon}"/>
            </button>
            <button class="sql-button sql-edit-btn" onclick="openInEditor(getSqlCode('${formId}'), '${messageId}')" 
                    data-form-id="${formId}" title="${editorTooltip}">
                <img class="sql-button-img" src="${editorIcon}"/>
            </button>
            <button class="sql-button sql-copy-btn" onclick="setClipboardContents(getSqlCode('${formId}'))" 
                    data-form-id="${formId}" title="${copyTooltip}">
                <img class="sql-button-img" src="${copyIcon}"/>
            </button>
        </div>
    `;
}


function getSqlCode(formId) {
    const form = document.getElementById(formId);
    if (form) {
        const codeLines = form.querySelectorAll('.code-line');
        const lines = Array.from(codeLines).map(line => {
            return line.textContent.replace(/\u00A0/g, ' ');
        });
        return lines.join('\n');
    }
    return '';
}

function decodeHtmlEntities(html) {
    const txt = document.createElement("textarea");
    txt.innerHTML = html;
    return txt.value;
}

function setBusy(args) {
    hideCenterText();

    const busy = args.busy;
    if (busy) {
        chat.appendChild(statusBar);
        chat.scrollTop = chat.scrollHeight;
    } else {
        statusBar.remove();
    }
}

function showCenterText(args) {
    hideCenterText();

    const centerDiv = document.createElement('div');
    centerDiv.id = 'center-text-content';
    centerDiv.innerHTML = `<div class="center-text-content">${args.text}</div>`;

    document.body.appendChild(centerDiv);
}

function hideCenterText() {
    const content = document.getElementById('center-text-content');
    if (content) {
        content.remove();
    }
}