CREATE TABLE QM_INFO (
    VERSION INTEGER NOT NULL,
    UPDATE_TIME TIMESTAMP NOT NULL
);

-- Deployment host information
CREATE TABLE QM_DBEAVER_HOST (
    HOST_ID INTEGER AUTO_INCREMENT NOT NULL,

    MAC_ADDRESS CHAR(12) NOT NULL,
    HOST_NAME VARCHAR(1024) NOT NULL,
    IP_ADDRESS VARCHAR(48) NOT NULL,

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (HOST_ID),
    UNIQUE (MAC_ADDRESS)
);

INSERT INTO QM_DBEAVER_HOST(HOST_ID, MAC_ADDRESS, HOST_NAME, IP_ADDRESS, UPDATE_TIME)
    VALUES (0, '000000000000', 'localhost', '127.0.0.1', CURRENT_TIMESTAMP);

-- User information
CREATE TABLE QM_USER (
    USER_ID INTEGER AUTO_INCREMENT NOT NULL,

    DOMAIN_NAME VARCHAR(128) NOT NULL,
    USER_NAME VARCHAR(255) NOT NULL,

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (USER_ID),
    UNIQUE (DOMAIN_NAME,USER_NAME)
);

-- Product deployment information
CREATE TABLE QM_DBEAVER_DEPLOYMENT (
    DEPLOY_ID INTEGER AUTO_INCREMENT NOT NULL,

    HOST_ID INTEGER NOT NULL DEFAULT 0,     -- Owner host
    INSTALL_PATH VARCHAR(1024) NOT NULL,    -- Local path on host machine
    PORT_NUMBER INTEGER DEFAULT 0,          -- Port number. 0 for desktop deployment

    HOST_NAME VARCHAR(128), -- Backward compatibility

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (DEPLOY_ID),
    UNIQUE (HOST_ID,INSTALL_PATH),

    FOREIGN KEY (HOST_ID) REFERENCES QM_DBEAVER_HOST(HOST_ID)
);

-- Product version dictionary
CREATE TABLE QM_DBEAVER_VERSION (
    VERSION_ID INTEGER AUTO_INCREMENT NOT NULL,

    PRODUCT_NAME VARCHAR(64) NOT NULL DEFAULT 'DBeaver',
    VERSION_NAME VARCHAR(32) NOT NULL,

    UPDATE_DEPLOY_ID INTEGER, -- Backward compatibility

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (VERSION_ID)
);

-- Workspace path dictionary
CREATE TABLE QM_DBEAVER_WORKSPACE (
    WORKSPACE_ID INTEGER AUTO_INCREMENT NOT NULL,

    HOST_ID INTEGER NOT NULL DEFAULT 0,
    WORKSPACE_PATH VARCHAR(1024) NOT NULL,      -- Workspace path on disk

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (WORKSPACE_ID),
    UNIQUE (HOST_ID,WORKSPACE_PATH),
    FOREIGN KEY (HOST_ID) REFERENCES QM_DBEAVER_HOST(HOST_ID)
);

CREATE TABLE QM_DBEAVER_PROJECT
(
    PROJECT_ID   INTEGER AUTO_INCREMENT NOT NULL,
    PROJECT_NAME VARCHAR(255) NOT NULL,
    PROJECT_UUID UUID NOT NULL,
    PROJECT_PATH VARCHAR(1024) NOT NULL,
    IS_ANONYMOUS BOOLEAN NULL,
    WORKSPACE_ID INTEGER       NOT NULL,
    UPDATE_TIME  TIMESTAMP     NOT NULL,

    PRIMARY KEY (PROJECT_ID),
    UNIQUE (WORKSPACE_ID, PROJECT_NAME),
    FOREIGN KEY (WORKSPACE_ID) REFERENCES QM_DBEAVER_WORKSPACE (WORKSPACE_ID)
);

-- Application run
CREATE TABLE QM_DBEAVER_RUN (
    RUN_ID BIGINT AUTO_INCREMENT NOT NULL,

    DEPLOY_ID INTEGER NOT NULL,
    VERSION_ID INTEGER NOT NULL,
    WORKSPACE_ID INTEGER NOT NULL,

    USER_ID INTEGER NOT NULL DEFAULT 0,     -- Owner user

    START_TIME TIMESTAMP NOT NULL,
    STOP_TIME TIMESTAMP,

    HOST_NAME VARCHAR(128),                 -- Backward compatibility

    PRIMARY KEY (RUN_ID),
    FOREIGN KEY (DEPLOY_ID) REFERENCES QM_DBEAVER_DEPLOYMENT(DEPLOY_ID),
    FOREIGN KEY (VERSION_ID) REFERENCES QM_DBEAVER_VERSION(VERSION_ID),
    FOREIGN KEY (WORKSPACE_ID) REFERENCES QM_DBEAVER_WORKSPACE(WORKSPACE_ID)
);

-- User session
CREATE TABLE QM_USER_SESSION (
    SESSION_ID BIGINT AUTO_INCREMENT NOT NULL,

    RUN_ID INTEGER NOT NULL,
    APP_SESSION_ID VARCHAR(64) NOT NULL,     -- Application-specific session ID. Unique within application run.

    USER_ID INTEGER NOT NULL,

    START_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (SESSION_ID),
    UNIQUE (RUN_ID,APP_SESSION_ID),
    FOREIGN KEY (USER_ID) REFERENCES QM_USER(USER_ID),
    FOREIGN KEY (RUN_ID) REFERENCES QM_DBEAVER_RUN(RUN_ID)
);

-- Connection configuration
CREATE TABLE QM_DATASOURCE (
    DATASOURCE_ID VARCHAR(100) NOT NULL,

    WORKSPACE_ID INTEGER NOT NULL DEFAULT 1,

    NAME VARCHAR(255) NOT NULL,
    DRIVER VARCHAR(255) NOT NULL,
    URL VARCHAR(1024) NOT NULL,
    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (DATASOURCE_ID),
    FOREIGN KEY (WORKSPACE_ID) REFERENCES QM_DBEAVER_WORKSPACE(WORKSPACE_ID)
);

-- Physical database connection info
CREATE TABLE QM_CONNECTION (
    CONNECTION_ID BIGINT AUTO_INCREMENT NOT NULL,

    RUN_ID BIGINT NOT NULL,
    SESSION_ID BIGINT,

    WORKSPACE_ID INTEGER NOT NULL DEFAULT 1,
    PROJECT_ID INTEGER,
    DATASOURCE_ID VARCHAR(100) NOT NULL,
    INSTANCE_ID VARCHAR(64),

    CONTEXT_ID VARCHAR(64),
    CONTEXT_NAME VARCHAR(255),
    CONNECTION_USER VARCHAR(64),

    CONNECT_TIME TIMESTAMP NOT NULL,
    DISCONNECT_TIME TIMESTAMP,

    PRIMARY KEY (CONNECTION_ID),
    FOREIGN KEY (RUN_ID) REFERENCES QM_DBEAVER_RUN(RUN_ID),
    FOREIGN KEY (SESSION_ID) REFERENCES QM_USER_SESSION(SESSION_ID),
    FOREIGN KEY (DATASOURCE_ID) REFERENCES QM_DATASOURCE(DATASOURCE_ID),
    FOREIGN KEY (PROJECT_ID) REFERENCES QM_DBEAVER_PROJECT(PROJECT_ID)
);

CREATE TABLE QM_EVENT_SOURCE (
    SOURCE_ID VARCHAR(64) NOT NULL,

    UPDATE_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (SOURCE_ID)
);

CREATE TABLE QM_EVENT (
    EVENT_ID BIGINT AUTO_INCREMENT NOT NULL,
    CONNECTION_ID BIGINT NOT NULL,
    SOURCE_ID VARCHAR(64),

    EVENT_ACTION INTEGER NOT NULL,
    EVENT_TYPE INTEGER NOT NULL,
    EVENT_TIME TIMESTAMP NOT NULL,

    PRIMARY KEY (EVENT_ID),
    FOREIGN KEY (CONNECTION_ID) REFERENCES QM_CONNECTION(CONNECTION_ID),
    FOREIGN KEY (SOURCE_ID) REFERENCES QM_EVENT_SOURCE(SOURCE_ID)
);

CREATE INDEX IDX_EVENT_ID_DESC ON QM_EVENT(EVENT_ID DESC);

CREATE TABLE QM_QUERY (
    QUERY_ID BIGINT AUTO_INCREMENT NOT NULL,
    QUERY_HASH BIGINT NOT NULL,

    QUERY_TEXT CLOB NOT NULL,

    PRIMARY KEY (QUERY_ID)
);

CREATE INDEX QM_QUERY_HASH_INDEX ON QM_QUERY(QUERY_HASH);

CREATE TABLE QM_QUERY_FILTER (
    FILTER_ID BIGINT AUTO_INCREMENT NOT NULL,
    QUERY_ID BIGINT NOT NULL,

    FILTER_HASH BIGINT NOT NULL,
    FILTER_TEXT CLOB NOT NULL,

    PRIMARY KEY (FILTER_ID),
    FOREIGN KEY (QUERY_ID) REFERENCES QM_QUERY(QUERY_ID)
);

CREATE INDEX QM_FILTER_HASH_INDEX ON QM_QUERY_FILTER(FILTER_HASH);

CREATE TABLE QM_STATEMENT (
    STATEMENT_ID BIGINT NOT NULL,

    STATEMENT_PURPOSE INTEGER NOT NULL,

    QUERY_ID BIGINT NOT NULL,
    ROW_COUNT INTEGER,
    ERROR_CODE INTEGER,
    ERROR_MESSAGE CLOB,

    EXECUTE_TIME INTEGER,
    FETCH_TIME INTEGER,

    PRIMARY KEY (STATEMENT_ID),
    FOREIGN KEY (STATEMENT_ID) REFERENCES QM_EVENT(EVENT_ID),
    FOREIGN KEY (QUERY_ID) REFERENCES QM_QUERY(QUERY_ID)
);

-- AI completion history

CREATE TABLE QM_AI_COMPLETION_HISTORY (
    COMPLETION_ID BIGINT AUTO_INCREMENT NOT NULL,
    USER_ID BIGINT NOT NULL,

    DATASOURCE_ID VARCHAR(100) NOT NULL,
    LOGICAL_DATASOURCE_NAME VARCHAR(100),
    DEFAULT_CATALOG VARCHAR(64) NOT NULL,
    DEFAULT_SCHEMA VARCHAR(64) NOT NULL,

    NATURAL_TEXT VARCHAR(16384) NOT NULL,
    SQL_TEXT CLOB NULL,

    FIRST_COMPLETION_TIME TIMESTAMP NOT NULL,
    LAST_COMPLETION_TIME TIMESTAMP NOT NULL,
    COMPLETION_COUNT INTEGER NOT NULL,

    COMPLETION_ERROR VARCHAR(2048),
    QUERY_ERROR VARCHAR(2048),

    PRIMARY KEY (USER_ID,COMPLETION_ID),
    FOREIGN KEY (USER_ID) REFERENCES QM_USER(USER_ID),
    FOREIGN KEY (DATASOURCE_ID) REFERENCES QM_DATASOURCE(DATASOURCE_ID)
);

CREATE INDEX IDX_AI_HISTORY_CONNECTION ON QM_AI_COMPLETION_HISTORY(USER_ID,DATASOURCE_ID,LOGICAL_DATASOURCE_NAME,DEFAULT_CATALOG,DEFAULT_SCHEMA,NATURAL_TEXT);
