/*
 * Decompiled with CFR 0.152.
 */
package org.jkiss.lm;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.jkiss.code.NotNull;
import org.jkiss.code.Nullable;
import org.jkiss.lm.LMException;
import org.jkiss.lm.LMKeyProvider;
import org.jkiss.lm.LMLicense;
import org.jkiss.lm.LMLicenseListener;
import org.jkiss.lm.LMLicenseStatus;
import org.jkiss.lm.LMLicenseType;
import org.jkiss.lm.LMLicenseValidator;
import org.jkiss.lm.LMProduct;
import org.jkiss.lm.LMStatusDetails;
import org.jkiss.lm.LMSubscription;
import org.jkiss.lm.LMUtils;
import org.jkiss.lm.LMValidateException;
import org.jkiss.utils.Base64;
import org.jkiss.utils.CommonUtils;
import org.jkiss.utils.IOUtils;
import org.jkiss.utils.xml.XMLBuilder;
import org.jkiss.utils.xml.XMLException;
import org.jkiss.utils.xml.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class LMLicenseManager {
    private static final Logger log = Logger.getLogger("LMLicenseManager");
    private static final String DEFAULT_PURCHASE_URL = "https://dbeaver.com/buy";
    private final LMKeyProvider keyProvider;
    private final LMLicenseValidator validator;
    private final Map<String, Map<String, LMLicense>> licenseCache = new HashMap<String, Map<String, LMLicense>>();
    private final Map<String, LMSubscription> subscriptionCache = new LinkedHashMap<String, LMSubscription>();
    private final List<LMLicenseListener> licenseListeners = new ArrayList<LMLicenseListener>();
    private File configPath;
    private static Path licensePath;
    private static final List<Path> licenseSearchPath;

    static {
        licenseSearchPath = new ArrayList<Path>();
    }

    public static Path getLicenseCustomPath() {
        return licensePath;
    }

    public static void setLicenseCustomPath(String path) {
        licensePath = Path.of(path, new String[0]);
    }

    public static void addLicenseSearchPath(Path file) {
        licenseSearchPath.add(file);
    }

    public LMLicenseManager(LMKeyProvider keyProvider, LMLicenseValidator validator) {
        this(keyProvider, validator, new File(System.getProperty("user.home"), ".jkiss-lm"));
    }

    public LMLicenseManager(LMKeyProvider keyProvider, LMLicenseValidator validator, File configPath) {
        this.keyProvider = keyProvider;
        this.validator = validator;
        this.configPath = configPath;
        if (licensePath == null && !configPath.exists() && !configPath.mkdirs()) {
            log.severe("Can't create LM config directory '" + configPath.getAbsolutePath() + "'");
        }
    }

    public File getConfigPath() {
        return this.configPath;
    }

    public void setConfigPath(File configPath) {
        this.configPath = configPath;
    }

    public static void setLicensePath(Path licensePath) {
        LMLicenseManager.licensePath = licensePath;
    }

    public void addLicenseListener(LMLicenseListener listener) {
        boolean add;
        if (listener == null) {
            log.severe("Ignored attempt to add null LMLicenseListener");
            return;
        }
        boolean remove = this.licenseListeners.remove(listener);
        if (remove) {
            log.warning(String.format("Removed existing LMLicenseListener %s", listener));
        }
        if (add = this.licenseListeners.add(listener)) {
            log.fine(String.format("Added LMLicenseListener %s", listener));
        } else {
            log.severe(String.format("Failed to add LMLicenseListener %s", listener));
        }
    }

    public void removeLicenseListener(LMLicenseListener listener) {
        if (listener == null) {
            log.severe("Ignored attempt to remove null LMLicenseListener");
            return;
        }
        boolean remove = this.licenseListeners.remove(listener);
        if (remove) {
            log.fine(String.format("Removed LMLicenseListener %s", listener));
        } else {
            log.warning(String.format("Nothing to remove for LMLicenseListener %s", listener));
        }
    }

    private void fireLicenseChanged(String id, LMLicense[] value) {
        LMLicense[] copy = Arrays.copyOf(value, value.length);
        for (LMLicenseListener listener : this.licenseListeners) {
            try {
                listener.licenseChanged(id, copy);
            }
            catch (Throwable t) {
                String pattern = "Error while processing licenseChanged for %s";
                String message = String.format(pattern, listener);
                log.log(Level.SEVERE, message, t);
            }
        }
    }

    @NotNull
    public LMLicense[] getProductLicenses(@NotNull LMProduct product) {
        String id = product.getId();
        Map<String, LMLicense> licenseMap = this.licenseCache.get(id);
        if (licenseMap != null) {
            return licenseMap.values().toArray(new LMLicense[0]);
        }
        LMLicense[] licenses = this.readProductLicenses(product);
        this.fireLicenseChanged(id, licenses);
        return licenses;
    }

    public LMLicense importLicense(@NotNull LMProduct product, @NotNull byte[] licenseData) throws LMException {
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        LMLicense license = new LMLicense(licenseData, decryptionKey);
        this.importLicense(product, license);
        return license;
    }

    public void importLicense(@NotNull LMProduct product, LMLicense license) throws LMException {
        if (license.isExpired()) {
            String exceptionMessageTemplate = "License %s is expired. You can purchase a new license on our website: %s.";
            String purchaseUrl = Platform.getProduct().getProperty("productPurchaseURL");
            if (CommonUtils.isEmpty((String)purchaseUrl)) {
                purchaseUrl = DEFAULT_PURCHASE_URL;
            }
            throw new LMException(String.format(exceptionMessageTemplate, license.getLicenseId(), purchaseUrl));
        }
        if (!license.isValidForProduct(product)) {
            String exceptionMessageTemplate = "License %s doesn't match product %s. If you want to access this DBeaver version, you need to extend your license support or purchase a new license.";
            throw new LMException(String.format(exceptionMessageTemplate, license.getProductId(), product.getId()));
        }
        if (!license.isValidFor(product, null, false)) {
            String exceptionMessageTemplate = "License %s is out of support. This product version is not available for your license. You need to extend the support and maintenance period to access this and upcoming DBeaver versions. Otherwise, you can download the previous versions from the archive";
            String archiveLink = Platform.getProduct().getProperty("archiveURL");
            exceptionMessageTemplate = CommonUtils.isEmpty((String)archiveLink) ? String.valueOf(exceptionMessageTemplate) + "." : String.valueOf(exceptionMessageTemplate) + ": " + archiveLink;
            throw new LMException(String.format(exceptionMessageTemplate, license.getLicenseId()));
        }
        LMLicense[] currentLicenses = this.getProductLicenses(product);
        LinkedHashMap<String, LMLicense> licenseMap = new LinkedHashMap<String, LMLicense>();
        LMLicense[] lMLicenseArray = currentLicenses;
        int n = currentLicenses.length;
        int n2 = 0;
        while (n2 < n) {
            LMLicense lic = lMLicenseArray[n2];
            licenseMap.put(lic.getLicenseId(), lic);
            ++n2;
        }
        if (license.getLicenseType() == LMLicenseType.TRIAL) {
            lMLicenseArray = currentLicenses;
            n = currentLicenses.length;
            n2 = 0;
            while (n2 < n) {
                LMLicense oldLic = lMLicenseArray[n2];
                if (oldLic.getLicenseType() == LMLicenseType.TRIAL && oldLic.getProductVersion().equalsIgnoreCase(license.getProductVersion())) {
                    String exceptionMessageTemplate = "You can't import a trial license for %s %s more than once.";
                    throw new LMException(String.format(exceptionMessageTemplate, license.getProductId(), license.getProductVersion()));
                }
                ++n2;
            }
        }
        licenseMap.put(license.getLicenseId(), license);
        this.licenseCache.put(product.getId(), licenseMap);
        this.saveProductLicenses(product);
        this.fireLicenseChanged(product.getId(), currentLicenses);
    }

    public void updateSubscription(@NotNull LMProduct product, @NotNull LMSubscription subscription) throws LMException {
        this.subscriptionCache.put(subscription.getLicenseId(), subscription);
        this.saveProductLicenses(product);
    }

    private void saveProductLicenses(LMProduct product) throws LMException {
        File prodLicenseFile = this.getProductLicensesFile(product);
        Map<String, LMLicense> cache = this.licenseCache.get(product.getId());
        if (cache == null || cache.isEmpty()) {
            if (prodLicenseFile.exists() && !prodLicenseFile.delete()) {
                log.warning("Can't delete licenses file '" + prodLicenseFile.getAbsolutePath() + "'");
            }
        } else {
            File lmDir = prodLicenseFile.getParentFile();
            if (!lmDir.exists() && !lmDir.mkdirs()) {
                log.warning("Can't create directory '" + lmDir.getAbsolutePath() + "'");
            }
            try {
                Throwable throwable = null;
                Object var6_8 = null;
                try (FileOutputStream out = new FileOutputStream(prodLicenseFile);){
                    XMLBuilder xml = new XMLBuilder((OutputStream)out, "utf-8");
                    xml.setButify(true);
                    Throwable throwable2 = null;
                    Object var10_14 = null;
                    try (XMLBuilder.Element el1 = xml.startElement("product");){
                        XMLBuilder.Element el2;
                        Object var16_23;
                        Throwable throwable3;
                        byte[] encodedData;
                        xml.addAttribute("id", product.getId());
                        for (LMLicense license : cache.values()) {
                            encodedData = license.getEncoded();
                            if (encodedData == null) {
                                log.warning("License '" + license.getLicenseId() + "' is not encoded");
                                continue;
                            }
                            throwable3 = null;
                            var16_23 = null;
                            try {
                                el2 = xml.startElement("license");
                                try {
                                    xml.addAttribute("type", "standard");
                                    xml.addText((CharSequence)Base64.encode((byte[])encodedData));
                                }
                                finally {
                                    if (el2 != null) {
                                        el2.close();
                                    }
                                }
                            }
                            catch (Throwable throwable4) {
                                if (throwable3 == null) {
                                    throwable3 = throwable4;
                                } else if (throwable3 != throwable4) {
                                    throwable3.addSuppressed(throwable4);
                                }
                                throw throwable3;
                            }
                        }
                        for (LMSubscription subscription : this.subscriptionCache.values()) {
                            encodedData = subscription.getEncoded();
                            if (encodedData == null) {
                                log.warning("Subscription '" + subscription.getLicenseId() + "' is not encoded");
                                continue;
                            }
                            throwable3 = null;
                            var16_23 = null;
                            try {
                                el2 = xml.startElement("subscription");
                                try {
                                    xml.addAttribute("type", "standard");
                                    xml.addText((CharSequence)Base64.encode((byte[])encodedData));
                                }
                                finally {
                                    if (el2 != null) {
                                        el2.close();
                                    }
                                }
                            }
                            catch (Throwable throwable5) {
                                if (throwable3 == null) {
                                    throwable3 = throwable5;
                                } else if (throwable3 != throwable5) {
                                    throwable3.addSuppressed(throwable5);
                                }
                                throw throwable3;
                            }
                        }
                    }
                    catch (Throwable throwable6) {
                        if (throwable2 == null) {
                            throwable2 = throwable6;
                        } else if (throwable2 != throwable6) {
                            throwable2.addSuppressed(throwable6);
                        }
                        throw throwable2;
                    }
                    xml.flush();
                }
                catch (Throwable throwable7) {
                    if (throwable == null) {
                        throwable = throwable7;
                    } else if (throwable != throwable7) {
                        throwable.addSuppressed(throwable7);
                    }
                    throw throwable;
                }
            }
            catch (IOException e) {
                throw new LMException("IO error while saving license file", e);
            }
        }
    }

    @NotNull
    private LMLicense[] readProductLicenses(@NotNull LMProduct product) {
        this.licenseCache.clear();
        this.subscriptionCache.clear();
        LinkedHashMap<String, LMLicense> licenses = new LinkedHashMap<String, LMLicense>();
        ArrayList<LMSubscription> subscriptions = new ArrayList<LMSubscription>();
        if (licensePath != null) {
            if (!Files.exists(licensePath, new LinkOption[0])) {
                log.warning("License file '" + licensePath.toAbsolutePath() + "' doesn't exist");
            } else {
                this.readLicenseFromFile(product, licensePath, licenses);
            }
        } else {
            for (Path licenseFile : licenseSearchPath) {
                if (!Files.exists(licenseFile, new LinkOption[0])) continue;
                this.readLicenseFromFile(product, licenseFile, licenses);
            }
        }
        File prodLicenseFile = this.getProductLicensesFile(product);
        if (prodLicenseFile.exists()) {
            try {
                String licenseType;
                Document document = XMLUtils.parseDocument((File)prodLicenseFile);
                Element rootElement = document.getDocumentElement();
                if (!"product".equals(rootElement.getTagName()) || !product.getId().equals(rootElement.getAttribute("id"))) {
                    throw new LMException("Bad license file structure");
                }
                for (Element licenseElement : XMLUtils.getChildElementList((Element)rootElement, (String)"license")) {
                    licenseType = licenseElement.getAttribute("type");
                    if (licenseType == null) {
                        log.warning("No license type");
                        continue;
                    }
                    String licenseEncoded = XMLUtils.getElementBody((Element)licenseElement);
                    if ("standard".equals(licenseType) && licenseEncoded != null) {
                        LMLicense license = this.readStandardLicense(product, licenseEncoded);
                        if (license == null) continue;
                        licenses.put(license.getLicenseId(), license);
                        continue;
                    }
                    log.warning("Unsupported license type: " + licenseType);
                }
                for (Element subElement : XMLUtils.getChildElementList((Element)rootElement, (String)"subscription")) {
                    licenseType = subElement.getAttribute("type");
                    if (licenseType == null) {
                        log.warning("No license type");
                        continue;
                    }
                    String subEncoded = XMLUtils.getElementBody((Element)subElement);
                    if ("standard".equals(licenseType) && subEncoded != null) {
                        LMSubscription subscription = this.readStandardSubscription(product, subEncoded);
                        if (subscription == null) continue;
                        subscriptions.add(subscription);
                        continue;
                    }
                    log.warning("Unsupported subscription type: " + licenseType);
                }
            }
            catch (LMException | XMLException e) {
                log.log(Level.SEVERE, "Error parse product license file '" + prodLicenseFile, e);
            }
        }
        this.licenseCache.put(product.getId(), licenses);
        for (LMSubscription subscription : subscriptions) {
            this.subscriptionCache.put(subscription.getLicenseId(), subscription);
        }
        return licenses.values().toArray(new LMLicense[0]);
    }

    private void readLicenseFromFile(@NotNull LMProduct product, @NotNull Path path, @NotNull Map<String, LMLicense> licenses) {
        try {
            Throwable throwable = null;
            Object var5_7 = null;
            try (BufferedReader in = Files.newBufferedReader(path);){
                String licenseEncoded = IOUtils.readToString((Reader)in);
                LMLicense license = this.readStandardLicense(product, licenseEncoded);
                if (license != null) {
                    licenses.put(license.getLicenseId(), license);
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            log.warning("Error loading custom license from " + path.toAbsolutePath() + ": " + e.getMessage());
        }
    }

    @NotNull
    private File getProductLicensesFile(@NotNull LMProduct product) {
        return new File(this.configPath, String.valueOf(product.getId()) + ".lic");
    }

    public LMLicense readStandardLicense(@NotNull LMProduct product, @NotNull String licenseEncoded) throws LMException {
        byte[] licenseEncrypted;
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        try {
            licenseEncrypted = LMUtils.readEncryptedString(new StringReader(licenseEncoded));
        }
        catch (IOException e) {
            log.log(Level.SEVERE, "Error reading license", e);
            return null;
        }
        try {
            return new LMLicense(licenseEncrypted, decryptionKey);
        }
        catch (LMException e) {
            log.log(Level.SEVERE, "Error parsing license", e);
            return null;
        }
    }

    private LMSubscription readStandardSubscription(@NotNull LMProduct product, @NotNull String subEncoded) throws LMException {
        byte[] subEncrypted;
        Key decryptionKey = this.keyProvider.getDecryptionKey(product);
        if (decryptionKey == null) {
            throw new LMException("Product '" + product.getId() + "' decryption key not found");
        }
        try {
            subEncrypted = LMUtils.readEncryptedString(new StringReader(subEncoded));
        }
        catch (IOException e) {
            log.log(Level.SEVERE, "Error reading subscription", e);
            return null;
        }
        try {
            return new LMSubscription(subEncrypted, decryptionKey);
        }
        catch (LMException e) {
            log.log(Level.SEVERE, "Error parsing subscription", e);
            return null;
        }
    }

    @Nullable
    public LMLicense getValidProductLicense(IProgressMonitor monitor, String clientId, @NotNull LMProduct product) throws LMValidateException {
        LMLicense[] licenses = this.getProductLicenses(product);
        if (licenses.length == 0) {
            return null;
        }
        LMValidateException validateError = null;
        ArrayList<LMLicense> validLicenses = new ArrayList<LMLicense>();
        LMLicense[] lMLicenseArray = licenses;
        int n = licenses.length;
        int n2 = 0;
        while (n2 < n) {
            block10: {
                LMLicense license = lMLicenseArray[n2];
                if (license.isValidFor(product, null, true)) {
                    if (clientId != null && this.validator != null && (validLicenses.isEmpty() || license.getLicenseType() != LMLicenseType.TRIAL)) {
                        try {
                            this.validator.validateLicense(monitor, this, clientId, product, license);
                        }
                        catch (LMValidateException e) {
                            validateError = e;
                            break block10;
                        }
                    }
                    validLicenses.add(license);
                }
            }
            ++n2;
        }
        if (validLicenses.isEmpty()) {
            if (validateError != null) {
                throw validateError;
            }
            return null;
        }
        validLicenses.sort(Comparator.comparing(LMLicense::getProductVersion).thenComparing(LMLicense::getLicenseStartTime));
        int i = validLicenses.size();
        while (i > 0) {
            LMLicense license = (LMLicense)validLicenses.get(i - 1);
            if (license.getLicenseType() != LMLicenseType.TRIAL) {
                return license;
            }
            --i;
        }
        return (LMLicense)validLicenses.get(validLicenses.size() - 1);
    }

    public boolean hasProductLicense(IProgressMonitor monitor, @NotNull LMProduct product) throws LMValidateException {
        return this.getValidProductLicense(monitor, null, product) != null;
    }

    public LMLicense findTrialLicense(IProgressMonitor monitor, String clientId, LMProduct product) throws LMValidateException {
        LMLicense[] licenses;
        LMValidateException validateError = null;
        LMLicense[] lMLicenseArray = licenses = this.getProductLicenses(product);
        int n = licenses.length;
        int n2 = 0;
        while (n2 < n) {
            block7: {
                LMLicense license = lMLicenseArray[n2];
                if (license.getLicenseType() == LMLicenseType.TRIAL && license.isValidFor(product, null, false)) {
                    if (this.validator != null) {
                        try {
                            this.validator.validateLicense(monitor, this, clientId, product, license);
                        }
                        catch (LMValidateException e) {
                            validateError = e;
                            break block7;
                        }
                    }
                    return license;
                }
            }
            ++n2;
        }
        if (validateError != null) {
            if (validateError.getStatus() == LMLicenseStatus.EXPIRED) {
                return null;
            }
            throw validateError;
        }
        return null;
    }

    public void deleteLicense(@NotNull LMProduct product, @NotNull LMLicense license) throws LMException {
        Map<String, LMLicense> licenseMap = this.licenseCache.get(product.getId());
        if (licenseMap == null) {
            throw new LMException("Internal error: product licenses not found");
        }
        licenseMap.remove(license.getLicenseId());
        this.subscriptionCache.remove(license.getLicenseId());
        this.validator.clearLicenseCache(product.getId());
        this.saveProductLicenses(product);
        this.fireLicenseChanged(product.getId(), licenseMap.values().toArray(new LMLicense[0]));
    }

    public void clearLicenseCache() {
        this.validator.clearLicenseCache();
    }

    public void validateLicense(IProgressMonitor monitor, String clientId, LMProduct product, LMLicense license) throws LMValidateException {
        if (this.validator != null) {
            this.validator.validateLicense(monitor, this, clientId, product, license);
        }
    }

    public LMSubscription getSubscriptionInfo(LMLicense license) {
        return this.subscriptionCache.get(license.getLicenseId());
    }

    public LMSubscription readSubscriptionFromData(@NotNull LMProduct product, byte[] encrypted) throws LMException {
        return new LMSubscription(encrypted, this.keyProvider.getDecryptionKey(product));
    }

    public String getLicenseValidationStatus(LMProduct product, LMLicense license) {
        String validationStatus;
        LMStatusDetails licenseStatus = license.getLicenseStatus(product);
        if (licenseStatus.isValid() && (validationStatus = this.validator.getLicenseValidationStatus(license)) != null) {
            return validationStatus;
        }
        return licenseStatus.getMessage();
    }
}

