/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore;

import java.io.IOException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplChangeManager {
    private static final Logger LOG = LoggerFactory.getLogger(ReplChangeManager.class);
    private static ReplChangeManager instance;
    private static boolean inited;
    private static boolean enabled;
    private static Path cmroot;
    private static HiveConf hiveConf;
    private String msUser;
    private String msGroup;
    private static final String ORIG_LOC_TAG = "user.original-loc";
    static final String REMAIN_IN_TRASH_TAG = "user.remain-in-trash";
    private static final String URI_FRAGMENT_SEPARATOR = "#";
    private static final PathFilter hiddenFileFilter;

    public static ReplChangeManager getInstance(HiveConf hiveConf) throws MetaException {
        if (instance == null) {
            instance = new ReplChangeManager(hiveConf);
        }
        return instance;
    }

    private ReplChangeManager(HiveConf hiveConf) throws MetaException {
        try {
            if (!inited) {
                if (hiveConf.getBoolVar(HiveConf.ConfVars.REPLCMENABLED)) {
                    enabled = true;
                    cmroot = new Path(hiveConf.get(HiveConf.ConfVars.REPLCMDIR.varname));
                    ReplChangeManager.hiveConf = hiveConf;
                    FileSystem cmFs = cmroot.getFileSystem(hiveConf);
                    if (!cmFs.exists(cmroot)) {
                        cmFs.mkdirs(cmroot);
                        cmFs.setPermission(cmroot, new FsPermission("700"));
                    }
                    UserGroupInformation usergroupInfo = UserGroupInformation.getCurrentUser();
                    this.msUser = usergroupInfo.getShortUserName();
                    this.msGroup = usergroupInfo.getPrimaryGroupName();
                }
                inited = true;
            }
        }
        catch (IOException e) {
            throw new MetaException(StringUtils.stringifyException(e));
        }
    }

    int recycle(Path path, RecycleType type, boolean ifPurge) throws MetaException {
        if (!enabled) {
            return 0;
        }
        try {
            int count = 0;
            FileSystem fs = path.getFileSystem(hiveConf);
            if (fs.isDirectory(path)) {
                FileStatus[] files;
                for (FileStatus file : files = fs.listStatus(path, hiddenFileFilter)) {
                    count += this.recycle(file.getPath(), type, ifPurge);
                }
            } else {
                String fileCheckSum = ReplChangeManager.checksumFor(path, fs);
                Path cmPath = ReplChangeManager.getCMPath(hiveConf, path.getName(), fileCheckSum, cmroot.toString());
                long now = System.currentTimeMillis();
                fs.setTimes(path, now, -1L);
                boolean success = false;
                if (fs.exists(cmPath) && fileCheckSum.equalsIgnoreCase(ReplChangeManager.checksumFor(cmPath, fs))) {
                    success = false;
                } else {
                    switch (type) {
                        case MOVE: {
                            LOG.info("Moving " + path.toString() + " to " + cmPath.toString());
                            success = fs.rename(path, cmPath);
                            break;
                        }
                        case COPY: {
                            LOG.info("Copying " + path.toString() + " to " + cmPath.toString());
                            success = FileUtils.copy(fs, path, fs, cmPath, false, true, hiveConf);
                            break;
                        }
                    }
                }
                if (success) {
                    fs.setOwner(cmPath, this.msUser, this.msGroup);
                    try {
                        fs.setXAttr(cmPath, ORIG_LOC_TAG, path.toString().getBytes());
                    }
                    catch (UnsupportedOperationException e) {
                        LOG.warn("Error setting xattr for " + path.toString());
                    }
                    ++count;
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("A file with the same content of " + path.toString() + " already exists, ignore");
                    }
                    fs.setTimes(cmPath, now, -1L);
                }
                if (type == RecycleType.MOVE && !ifPurge) {
                    try {
                        fs.setXAttr(cmPath, REMAIN_IN_TRASH_TAG, new byte[]{0});
                    }
                    catch (UnsupportedOperationException e) {
                        LOG.warn("Error setting xattr for " + cmPath.toString());
                    }
                }
            }
            return count;
        }
        catch (IOException e) {
            throw new MetaException(StringUtils.stringifyException(e));
        }
    }

    public static String checksumFor(Path path, FileSystem fs) throws IOException {
        String checksumString = null;
        FileChecksum checksum = fs.getFileChecksum(path);
        if (checksum != null) {
            checksumString = StringUtils.byteToHexString(checksum.getBytes(), 0, checksum.getLength());
        }
        return checksumString;
    }

    static Path getCMPath(Configuration conf, String name, String checkSum, String cmRootUri) throws IOException, MetaException {
        String newFileName = name + "_" + checkSum;
        int maxLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 255);
        if (newFileName.length() > maxLength) {
            newFileName = newFileName.substring(0, maxLength - 1);
        }
        return new Path(cmRootUri, newFileName);
    }

    public static FileInfo getFileInfo(Path src, String checksumString, String srcCMRootURI, HiveConf hiveConf) throws MetaException {
        try {
            String currentChecksumString;
            FileSystem srcFs = src.getFileSystem(hiveConf);
            if (checksumString == null) {
                return new FileInfo(srcFs, src);
            }
            Path cmPath = ReplChangeManager.getCMPath(hiveConf, src.getName(), checksumString, srcCMRootURI);
            if (!srcFs.exists(src)) {
                return new FileInfo(srcFs, src, cmPath, checksumString, false);
            }
            try {
                currentChecksumString = ReplChangeManager.checksumFor(src, srcFs);
            }
            catch (IOException ex) {
                return new FileInfo(srcFs, src, cmPath, checksumString, false);
            }
            if (currentChecksumString == null || checksumString.equals(currentChecksumString)) {
                return new FileInfo(srcFs, src, cmPath, checksumString, true);
            }
            return new FileInfo(srcFs, src, cmPath, checksumString, false);
        }
        catch (IOException e) {
            throw new MetaException(StringUtils.stringifyException(e));
        }
    }

    public static String encodeFileUri(String fileUriStr, String fileChecksum) throws IOException {
        if (fileChecksum != null && cmroot != null) {
            String encodedUri = fileUriStr + URI_FRAGMENT_SEPARATOR + fileChecksum + URI_FRAGMENT_SEPARATOR + FileUtils.makeQualified(cmroot, hiveConf);
            LOG.debug("Encoded URI: " + encodedUri);
            return encodedUri;
        }
        return fileUriStr;
    }

    public static String[] getFileWithChksumAndCmRootFromURI(String fileURIStr) {
        String[] uriAndFragment = fileURIStr.split(URI_FRAGMENT_SEPARATOR);
        String[] result = new String[3];
        result[0] = uriAndFragment[0];
        if (uriAndFragment.length > 1) {
            result[1] = uriAndFragment[1];
        }
        if (uriAndFragment.length > 2) {
            result[2] = uriAndFragment[2];
        }
        LOG.debug("Reading Encoded URI: " + result[0] + ":: " + result[1] + ":: " + result[2]);
        return result;
    }

    public static boolean isCMFileUri(Path fromPath, FileSystem srcFs) {
        String[] result = ReplChangeManager.getFileWithChksumAndCmRootFromURI(fromPath.toString());
        return result[1] != null;
    }

    static void scheduleCMClearer(HiveConf hiveConf) {
        if (HiveConf.getBoolVar(hiveConf, HiveConf.ConfVars.REPLCMENABLED)) {
            ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor((ThreadFactory)new BasicThreadFactory.Builder().namingPattern("cmclearer-%d").daemon(true).build());
            executor.scheduleAtFixedRate(new CMClearer(hiveConf.get(HiveConf.ConfVars.REPLCMDIR.varname), hiveConf.getTimeVar(HiveConf.ConfVars.REPLCMRETIAN, TimeUnit.SECONDS), hiveConf), 0L, hiveConf.getTimeVar(HiveConf.ConfVars.REPLCMINTERVAL, TimeUnit.SECONDS), TimeUnit.SECONDS);
        }
    }

    static {
        inited = false;
        enabled = false;
        hiddenFileFilter = new PathFilter(){

            @Override
            public boolean accept(Path p) {
                return !p.getName().startsWith(".");
            }
        };
    }

    static class CMClearer
    implements Runnable {
        private Path cmroot;
        private long secRetain;
        private HiveConf hiveConf;

        CMClearer(String cmrootString, long secRetain, HiveConf hiveConf) {
            this.cmroot = new Path(cmrootString);
            this.secRetain = secRetain;
            this.hiveConf = hiveConf;
        }

        @Override
        public void run() {
            try {
                FileStatus[] files;
                LOG.info("CMClearer started");
                long now = System.currentTimeMillis();
                FileSystem fs = this.cmroot.getFileSystem(this.hiveConf);
                for (FileStatus file : files = fs.listStatus(this.cmroot)) {
                    long modifiedTime = file.getModificationTime();
                    if (now - modifiedTime <= this.secRetain * 1000L) continue;
                    try {
                        boolean succ;
                        if (fs.getXAttrs(file.getPath()).containsKey(ReplChangeManager.REMAIN_IN_TRASH_TAG)) {
                            succ = Trash.moveToAppropriateTrash(fs, file.getPath(), this.hiveConf);
                            if (succ) {
                                if (!LOG.isDebugEnabled()) continue;
                                LOG.debug("Move " + file.toString() + " to trash");
                                continue;
                            }
                            LOG.warn("Fail to move " + file.toString() + " to trash");
                            continue;
                        }
                        succ = fs.delete(file.getPath(), false);
                        if (succ) {
                            if (!LOG.isDebugEnabled()) continue;
                            LOG.debug("Remove " + file.toString());
                            continue;
                        }
                        LOG.warn("Fail to remove " + file.toString());
                    }
                    catch (UnsupportedOperationException e) {
                        LOG.warn("Error getting xattr for " + file.getPath().toString());
                    }
                }
            }
            catch (IOException e) {
                LOG.error("Exception when clearing cmroot:" + StringUtils.stringifyException(e));
            }
        }
    }

    public static class FileInfo {
        FileSystem srcFs;
        Path sourcePath;
        Path cmPath;
        String checkSum;
        boolean useSourcePath;
        boolean copyDone;

        public FileInfo(FileSystem srcFs, Path sourcePath) {
            this(srcFs, sourcePath, null, null, true);
        }

        public FileInfo(FileSystem srcFs, Path sourcePath, Path cmPath, String checkSum, boolean useSourcePath) {
            this.srcFs = srcFs;
            this.sourcePath = sourcePath;
            this.cmPath = cmPath;
            this.checkSum = checkSum;
            this.useSourcePath = useSourcePath;
            this.copyDone = false;
        }

        public FileSystem getSrcFs() {
            return this.srcFs;
        }

        public Path getSourcePath() {
            return this.sourcePath;
        }

        public Path getCmPath() {
            return this.cmPath;
        }

        public String getCheckSum() {
            return this.checkSum;
        }

        public boolean isUseSourcePath() {
            return this.useSourcePath;
        }

        public void setUseSourcePath(boolean useSourcePath) {
            this.useSourcePath = useSourcePath;
        }

        public boolean isCopyDone() {
            return this.copyDone;
        }

        public void setCopyDone(boolean copyDone) {
            this.copyDone = copyDone;
        }

        public Path getEffectivePath() {
            if (this.useSourcePath) {
                return this.sourcePath;
            }
            return this.cmPath;
        }
    }

    public static enum RecycleType {
        MOVE,
        COPY;

    }
}

